/* $Id: cmd_put.c,v 1.5 2003/12/13 21:20:05 short Exp $
 * client cmdline interface command "put" for libcaptive
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include <glib/gmessages.h>
#include <glib/gerror.h>
#include <popt.h>
#include <captive/client-file.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <captive/macros.h>
#include <stdlib.h>

#include "cmd_put.h"	/* self */
#include "cmd_cd.h"	/* for cmdline_path_from_cwd() */
#include "main.h"
#include "utf8.h"


/* Config: */
#define DEFAULT_TRANSFER_BUFFER_SIZE (0x10000)


GQuark cmdline_cmd_put_error_quark(void)
{
GQuark r=0;

	if (!r)
		r=g_quark_from_static_string("cmdline-cmd-put");

	return r;
}


static gchar *optarg_transfer_buffer;

const struct poptOption cmd_put_table[]={
		CMDLINE_POPT("buffer-size",'b',POPT_ARG_STRING,&optarg_transfer_buffer,
				N_("Transfer buffer size"),N_("size")),
		CMDLINE_POPT_AUTOHELP
		POPT_TABLEEND
		};


void cmd_put(const char **cmd_argv,GError **errp)
{
CaptiveFileObject *captive_file_object;
const gchar *sourcefile,*targetfile;
guint perm=0644;
int fdsrc;
guint8 *transfer_buffer;
long transfer_buffer_size=DEFAULT_TRANSFER_BUFFER_SIZE;
int got;

	g_return_if_fail(!errp || !*errp);

	if (optarg_transfer_buffer) {
const gchar *string=captive_strdup_alloca(optarg_transfer_buffer);
char *endptr;

		free(optarg_transfer_buffer);
		optarg_transfer_buffer=NULL;

		transfer_buffer_size=strtol(string,&endptr,0);
		if (transfer_buffer_size<=0 || transfer_buffer_size>=LONG_MAX || (endptr && *endptr)) {
			g_set_error(errp,CMDLINE_CMD_PUT_ERROR,CMDLINE_CMD_PUT_ERROR_PARSING_TRANSFER_BUFFER_SIZE,
					_("Error parsing transfer buffer size: %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(string));
			return;
			}
		}

	sourcefile=cmd_argv[0];
	if (cmd_argv[1])
		targetfile=cmdline_path_from_cwd(cmd_argv[1]);
	else {
gchar *sourcefile_basename;

		sourcefile_basename=g_path_get_basename(sourcefile);
		targetfile=cmdline_path_from_cwd(sourcefile_basename);
		g_free(sourcefile_basename);
		}

	if (-1==(fdsrc=open(CMD_FILENAME_FROM_UTF8_ALLOCA(sourcefile),O_RDONLY
#ifdef O_BINARY
			| O_BINARY
#endif /* O_BINARY */
#ifdef O_LARGEFILE
			| O_LARGEFILE
#endif /* O_LARGEFILE */
			))) {
		g_set_error(errp,CMDLINE_CMD_PUT_ERROR,CMDLINE_CMD_PUT_ERROR_OPENING_SOURCE_FILE,
				_("Error opening source host-os file '%s': %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(sourcefile),g_strerror(errno));
		return;
		}

	if (!errvfsresult_to_gerr(errp,captive_file_new_create(
			&captive_file_object,	/* captive_file_object_return */
			cmdline_captive_vfs_object,	/* captive_vfs_object */
			targetfile,	/* pathname */
			GNOME_VFS_OPEN_WRITE,	/* mode */
			FALSE,	/* exclusive */
			perm))) {	/* perm */
		err_cleanup(errp);
		g_set_error(errp,CMDLINE_CMD_PUT_ERROR,CMDLINE_CMD_PUT_ERROR_CANNOT_CREATE_TARGET_GUESTOS_FILE,
				_("Cannot create target guest-os file: %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(targetfile));
		goto err_close_fdsrc;
		}

	transfer_buffer=g_malloc(transfer_buffer_size);
	g_assert(transfer_buffer!=NULL);	/* Should not happen. */

	while (0<(got=read(fdsrc,transfer_buffer,transfer_buffer_size))) {
GnomeVFSFileSize bytes_written;

		if (!errvfsresult_to_gerr(errp,captive_file_write(
				captive_file_object,	/* captive_file_object */
				transfer_buffer,	/* buffer */
				got,	/* num_bytes */
				&bytes_written))) {	/* bytes_written_return */
			err_cleanup(errp);
			g_set_error(errp,CMDLINE_CMD_PUT_ERROR,CMDLINE_CMD_PUT_ERROR_WRITING_TARGET_GUESTOS_FILE,
					_("Error writing target guest-os file: %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(targetfile));
			goto err_free_transfer_buffer;
			}
		if (bytes_written!=(GnomeVFSFileSize)got) {
			g_set_error(errp,CMDLINE_CMD_PUT_ERROR,CMDLINE_CMD_PUT_ERROR_WRITING_TARGET_GUESTOS_FILE,
					_("Error writing target guest-os file '%s': requested %d, written %Lu"),
					CMD_LOCALE_FROM_UTF8_ALLOCA(targetfile),got,(unsigned long long)bytes_written);
			goto err_free_transfer_buffer;
			}
		}
	if (got==-1) {
		g_set_error(errp,CMDLINE_CMD_PUT_ERROR,CMDLINE_CMD_PUT_ERROR_READING_SOURCE_FILE,
				_("Error reading source host-os file '%s': %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(sourcefile),g_strerror(errno));
		goto err_free_transfer_buffer;
		}
	g_assert(got==0);

err_free_transfer_buffer:
	g_free(transfer_buffer);
/* err_unref_captive_file_object: */
	g_object_unref(captive_file_object);
err_close_fdsrc:
	if (close(fdsrc)) {
		err_cleanup(errp);	/* may be clean */
		g_set_error(errp,CMDLINE_CMD_PUT_ERROR,CMDLINE_CMD_PUT_ERROR_CLOSING_SOURCE_FILE,
				_("Error closing source host-os file '%s': %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(sourcefile),g_strerror(errno));
		}
}
