/* $Id: file_info.c,v 1.5 2006/01/01 07:24:34 lace Exp $
 * client cmdline interface GnomeVFSFileInfo utils for libcaptive
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include <glib/gmessages.h>
#include <glib/gerror.h>
#include <libgnomevfs/gnome-vfs-file-info.h>
#include <glib/gstrfuncs.h>
#include <stdio.h>
#include <time.h>


#include "file_info.h"	/* self */
#include "utf8.h"


GQuark cmdline_file_info_error_quark(void)
{
GQuark r=0;

	if (!r)
		r=g_quark_from_static_string("cmdline-file_info");

	return r;
}


void cmdline_captive_file_info_object_dump_line(CaptiveFileInfoObject *captive_file_info_object,GError **errp)
{
const gchar *file_type,*file_perms;
gchar *file_size;

	g_return_if_fail(CAPTIVE_FILE_INFO_IS_OBJECT(captive_file_info_object));
	g_return_if_fail(!errp || !*errp);

	switch (!(captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_TYPE) ? GNOME_VFS_FILE_TYPE_UNKNOWN : captive_file_info_object->p.type) {
		case GNOME_VFS_FILE_TYPE_REGULAR:   file_type="FILE"; break;
		case GNOME_VFS_FILE_TYPE_DIRECTORY: file_type="DIR ";  break;
		case GNOME_VFS_FILE_TYPE_SOCKET:    file_type="DEV ";  break;
		default:                            file_type="??? ";  break;
		}

	if (!(captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS))
		file_perms="???";
	else if (captive_file_info_object->p.permissions & GNOME_VFS_PERM_USER_WRITE)
		file_perms="r/w";
	else
		file_perms="r/o";

	if (captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SIZE)
		file_size=g_strdup_printf("%8" GNOME_VFS_SIZE_FORMAT_STR,captive_file_info_object->p.size);
	else
		file_size=g_strdup_printf("%8s","???");

	/*      type pm sz nm */
	printf("[%s] %s %s %s\n",file_type,file_perms,file_size,CMD_LOCALE_FROM_UTF8_ALLOCA(captive_file_info_object->p.name));
	
	g_free(file_size);
}

static void timespec_split_print(time_t sec,guint nsec)
{
char *ctime_buf,*s;

	g_return_if_fail(sec!=0);
	g_return_if_fail(nsec<1000000000);

	ctime_buf=ctime(&sec);
	if ((s=strchr(ctime_buf,'\n')))
		*s='\0';
	printf("%s +%09u nsec\n",ctime_buf,(unsigned)nsec);
}

void cmdline_captive_file_info_object_dump_full(CaptiveFileInfoObject *captive_file_info_object,GError **errp)
{
	g_return_if_fail(CAPTIVE_FILE_INFO_IS_OBJECT(captive_file_info_object));
	g_return_if_fail(!errp || !*errp);

	printf("Filename: %s\n",CMD_LOCALE_FROM_UTF8_ALLOCA(captive_file_info_object->p.name));

	fputs("File type: ",stdout);
	switch (!(captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_TYPE) ? GNOME_VFS_FILE_TYPE_UNKNOWN : captive_file_info_object->p.type) {
		case GNOME_VFS_FILE_TYPE_REGULAR:   puts("REGULAR");   break;
		case GNOME_VFS_FILE_TYPE_DIRECTORY: puts("DIRECTORY"); break;
		case GNOME_VFS_FILE_TYPE_SOCKET:    puts("SOCKET");    break;
		default:                            puts("???");       break;
		}

	fputs("File size: ",stdout);
	if (captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SIZE)
		printf("%" GNOME_VFS_SIZE_FORMAT_STR "\n",captive_file_info_object->p.size);
	else
		puts("???");

	fputs("Block count: ",stdout);
	if (captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_BLOCK_COUNT)
		printf("%" GNOME_VFS_SIZE_FORMAT_STR "\n",captive_file_info_object->p.block_count);
	else
		puts("???");

	fputs("Writable?: ",stdout);
	if (!(captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS))
		puts("???");
	else if (captive_file_info_object->p.permissions & GNOME_VFS_PERM_USER_WRITE)
		puts("read/write");
	else
		puts("read/only");

	fputs("Access-time (atime): ",stdout);
	if (!(captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_ATIME))
		puts("???");
	else
		timespec_split_print(captive_file_info_object->p.atime,captive_file_info_object->atime_nsec);

	fputs("Modification-time (mtime): ",stdout);
	if (!(captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MTIME))
		puts("???");
	else
		timespec_split_print(captive_file_info_object->p.mtime,captive_file_info_object->mtime_nsec);

	fputs("Change-time (ctime): ",stdout);
	if (!(captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_CTIME))
		puts("???");
	else
		timespec_split_print(captive_file_info_object->p.ctime,captive_file_info_object->ctime_nsec);
}
