/* $Id: handle.c,v 1.2 2003/12/13 21:20:05 short Exp $
 * client cmdline named CaptiveFileObject storage for libcaptive
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include <glib/gmessages.h>
#include <glib/gerror.h>
#include <captive/client-file.h>

#include "handle.h"	/* self */
#include "utf8.h"


GQuark cmdline_handle_error_quark(void)
{
GQuark r=0;

	if (!r)
		r=g_quark_from_static_string("cmdline-handle");

	return r;
}


static GHashTable *handle_hash;

static void handle_hash_key_destroy_func(gchar *handle_name /* key */)
{
	g_return_if_fail(handle_name!=NULL);

	g_free(handle_name);
}

static void handle_hash_value_destroy_func(CaptiveFileObject *captive_file_object /* value */)
{
	g_return_if_fail(CAPTIVE_FILE_IS_OBJECT(captive_file_object));

	g_object_unref(captive_file_object);
}

static void handle_hash_init(void)
{
	if (handle_hash)
		return;
	handle_hash=g_hash_table_new_full(
			g_str_hash,	/* hash_func */
			g_str_equal,	/* key_equal_func */
			(GDestroyNotify)handle_hash_key_destroy_func,	/* key_destroy_func */
			(GDestroyNotify)handle_hash_value_destroy_func);	/* value_destroy_func */
}

CaptiveFileObject *handle_get(const gchar *handle_name,GError **errp)
{
CaptiveFileObject *r;

	g_return_val_if_fail(handle_name!=NULL,NULL);
	g_return_val_if_fail(!errp || !*errp,NULL);

	handle_hash_init();

	if (!(r=g_hash_table_lookup(handle_hash,handle_name)))
		g_set_error(errp,CMDLINE_HANDLE_ERROR,CMDLINE_HANDLE_ERROR_LOOKUP,
				_("Error looking up existing handle: %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(handle_name));

	return r;
}

/* Returns: TRUE if not used. */
gboolean handle_check_not_used(const gchar *handle_name,GError **errp)
{
gboolean r;

	g_return_val_if_fail(handle_name!=NULL,FALSE);
	g_return_val_if_fail(!errp || !*errp,FALSE);

	handle_hash_init();

	if (!(r=!g_hash_table_lookup(handle_hash,handle_name)))
		g_set_error(errp,CMDLINE_HANDLE_ERROR,CMDLINE_HANDLE_ERROR_ALREADY_USED,
				_("Specified handle already used: %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(handle_name));

	return r;
}

void handle_set(const gchar *handle_name,CaptiveFileObject *captive_file_object)
{
	g_return_if_fail(handle_name!=NULL);
	g_return_if_fail(CAPTIVE_FILE_IS_OBJECT(captive_file_object));

	handle_hash_init();

	g_return_if_fail(NULL==g_hash_table_lookup(handle_hash,handle_name));

	g_hash_table_insert(handle_hash,
			g_strdup(handle_name),	/* key */
			g_object_ref(captive_file_object));	/* value */
}

void handle_delete(const gchar *handle_name,GError **errp)
{
	g_return_if_fail(handle_name!=NULL);
	g_return_if_fail(!errp || !*errp);

	handle_hash_init();

	if (!g_hash_table_remove(handle_hash,handle_name)) {
		g_set_error(errp,CMDLINE_HANDLE_ERROR,CMDLINE_HANDLE_ERROR_DELETE,
				_("Handle to be deleted not found: %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(handle_name));
		return;
		}

	g_assert(NULL==g_hash_table_lookup(handle_hash,handle_name));
}
