/* $Id: main.c,v 1.24 2005/12/22 08:22:56 lace Exp $
 * client cmdline interface for libcaptive
 * Copyright (C) 2002-2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include <glib/gmessages.h>
#include <stdlib.h>
#include <glib/giochannel.h>
#include <glib/gerror.h>
#include <popt.h>
#include <string.h>
#include <stdio.h>
#include <locale.h>

#include <captive/client-vfs.h>
#include <captive/client.h>

#include "main.h"	/* self */
#include "cmd_shell.h"
#include "cmd_cd.h"
#include "cmd_lcd.h"
#include "cmd_ls.h"
#include "cmd_get.h"
#include "cmd_put.h"
#include "cmd_info.h"
#include "cmd_volume.h"
#include "cmd_rm.h"
#include "cmd_mv.h"
#include "cmd_mkdir.h"
#include "cmd_rmdir.h"
#include "cmd_commit.h"
#include "cmd_open.h"
#include "cmd_create.h"
#include "cmd_close.h"
#include "cmd_quit.h"
#include "cmd_help.h"
#include "utf8.h"


CaptiveVfsObject *cmdline_captive_vfs_object;


GQuark cmdline_main_error_quark(void)
{
GQuark r=0;

	if (!r)
		r=g_quark_from_static_string("cmdline-main");

	return r;
}


static const struct poptOption popt_table[]={
		CAPTIVE_POPT_INCLUDE,
		POPT_AUTOHELP
		POPT_TABLEEND
		};

const struct cmdline_command cmdline_command_table[]={
		/* First entry is the default if no command name was specified. */
		{ "shell" ,N_("Interactive commands shell.")                                ,cmd_shell_table ,cmd_shell ,0,0 },
		{ "cd"    ,N_("Print or change current guest-os directory[1].")             ,cmd_cd_table    ,cmd_cd    ,0,1 },
		{ "lcd"   ,N_("Print or change current host-os  directory[1].")             ,cmd_lcd_table   ,cmd_lcd   ,0,1 },
		{ "ls"    ,N_("Directory[1] listing.")                                      ,cmd_ls_table    ,cmd_ls    ,0,1 },
		{ "get"   ,N_("Copy guest-os file[1] to host-os (opt. file[2]).")           ,cmd_get_table   ,cmd_get   ,1,2 },
		{ "put"   ,N_("Copy host-os file[1] to guest-os (opt. file[2]).")           ,cmd_put_table   ,cmd_put   ,1,2 },
		{ "info"  ,N_("Query information about guest-os item[1].")                  ,cmd_info_table  ,cmd_info  ,1,1 },
		{ "volume",N_("Query information about guest-os volume.")                   ,cmd_volume_table,cmd_volume,0,0 },
		{ "rm"    ,N_("Remove guest-os file[1].")                                   ,cmd_rm_table    ,cmd_rm    ,1,1 },
		{ "mv"    ,N_("Move (rename) guest-os item[1] to guest-os item[2].")        ,cmd_mv_table    ,cmd_mv    ,2,2 },
		{ "mkdir" ,N_("Create guest-os directory[1].")                              ,cmd_mkdir_table ,cmd_mkdir ,1,1 },
		{ "rmdir" ,N_("Remove guest-os directory[1].")                              ,cmd_rmdir_table ,cmd_rmdir ,1,1 },
		{ "commit",N_("Write any pending changes and remount the volume.")          ,cmd_commit_table,cmd_commit,0,0 },
		{ "open"  ,N_("Open as[1] file[2] in mode; see 'open --help'")              ,cmd_open_table  ,cmd_open  ,2,2 },
		{ "create",N_("Create as[1] file[2] in mode with perm; see 'create --help'"),cmd_create_table,cmd_create,2,2 },
		{ "close" ,N_("Close handle[1]")                                            ,cmd_close_table ,cmd_close ,1,1 },
		{ "quit"  ,N_("Quit this program.")                                         ,cmd_quit_table  ,cmd_quit  ,0,0 },
		{ "help"  ,N_("Show this list of commands or help for command[1].")         ,cmd_help_table  ,cmd_help  ,0,1 },
		{ NULL },	/* G_N_ELEMENTS() not usable as sizeof() is not visible for 'extern' */
		};


static gboolean displayArgs_hit;

static void displayArgs(poptContext con,enum poptCallbackReason foo,struct poptOption *key,const char *arg,void *data)
{
	displayArgs_hit=TRUE;

	if (key->shortName=='?')
		poptPrintHelp(con,stdout,0);
	else
		poptPrintUsage(con,stdout,0);
}

const struct poptOption cmdline_poptHelpOptions[]={
		{ argInfo:POPT_ARG_INTL_DOMAIN,arg:"popt" },
		{ NULL   ,'\0',POPT_ARG_CALLBACK,(void *)&displayArgs,'\0',NULL,NULL },
		{ "help" ,'?' ,0                ,NULL,'?',/* N_ */("Show this help message"),     NULL },
		{ "usage",'\0',0                ,NULL,'u',/* N_ */("Display brief usage message"),NULL },
		POPT_TABLEEND
		};


static void invoke_cmd_err(int cmd_argc,const char **cmd_argv,GError **errp)
{
const struct cmdline_command *commandp;
const char *cmd_name=NULL;
poptContext cmd_context;
int errint;
const char **cmdarg_argv;
int cmdarg_argc,argci;
const char **csp,*cs;
const char *emptyargv_NULL=NULL;

	g_return_if_fail(cmd_argc>=0);
	g_return_if_fail(!errp || !*errp);

	/* poptGetContext() cannot be passed argc==0 even if we lass POPT_CONTEXT_KEEP_FIRST
	 * as it is buggy. Workaround it by keeping the command name as argv[0].
	 */
	if (!cmd_argc) {
const char *stub_shell[]={ cmdline_command_table[0].name,NULL };
		
		cmd_argc=1;
		cmd_argv=stub_shell;
		}

	for (argci=0;argci<cmd_argc;argci++) {
		if ((cs=CMD_LOCALE_TO_UTF8_ALLOCA(cmd_argv[argci])))
			cmd_argv[argci]=cs;
		}

	cmd_name=*cmd_argv;
	for (commandp=cmdline_command_table;commandp->name;commandp++) {
		if (!cmd_name	/* NULL cmd_name fallback to the first table entry - "shell" */
				|| !strcasecmp(cmd_name,commandp->name))
			break;
		}
	if (!commandp->name) {
		g_set_error(errp,CMDLINE_MAIN_ERROR,CMDLINE_MAIN_ERROR_UNKNOWN_COMMAND,
				_("Unknown command, try 'help': %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(cmd_name));
		return;
		}
	displayArgs_hit=FALSE;
	cmd_context=poptGetContext(
			PACKAGE,	/* name */
			cmd_argc,cmd_argv,	/* argc,argv */
			commandp->table,	/* options */
			POPT_CONTEXT_POSIXMEHARDER);	/* flags; !POPT_CONTEXT_KEEP_FIRST */
	if (cmd_context==NULL) {
		g_set_error(errp,CMDLINE_MAIN_ERROR,CMDLINE_MAIN_ERROR_INVALID_COMMAND_ARGUMENTS,
				_("Invalid arguments for command: %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(cmd_name));
		return;
		}
	errint=poptReadDefaultConfig(cmd_context,
			TRUE);	/* useEnv */
	if (errint!=0) {
		g_set_error(errp,CMDLINE_MAIN_ERROR,CMDLINE_MAIN_ERROR_READING_COMMAND_CONFIG,
				_("Error '%s' reading default configuration for command: %s"),
				poptStrerror(errint),CMD_LOCALE_FROM_UTF8_ALLOCA(cmd_name));
		goto err_free_context;
		}
	errint=poptGetNextOpt(cmd_context);
	if (errint!=-1) {
		g_set_error(errp,CMDLINE_MAIN_ERROR,CMDLINE_MAIN_ERROR_EXCEEDING_COMMAND_OPTION,
				_("Exceeding command option for command: %s"),CMD_LOCALE_FROM_UTF8_ALLOCA(cmd_name));
		goto err_free_context;
		}
	if (!(cmdarg_argv=poptGetArgs(cmd_context)))
		cmdarg_argv=&emptyargv_NULL;

	if (displayArgs_hit)
		goto err_free_context;
	
	for (csp=cmdarg_argv,cmdarg_argc=0;*csp;csp++)
		cmdarg_argc++;
	if (cmdarg_argc<commandp->argsn_min || cmdarg_argc>commandp->argsn_max) {
		g_set_error(errp,CMDLINE_MAIN_ERROR,CMDLINE_MAIN_ERROR_INVALID_COMMAND_ARGUMENT_COUNT,
				_("Invalid number of command '%s' arguments: %d; expected from %d to %d incl."),
				CMD_LOCALE_FROM_UTF8_ALLOCA(cmd_name),cmdarg_argc,commandp->argsn_min,commandp->argsn_max);
		goto err_free_context;
		}

	(*commandp->func)(cmdarg_argv,errp);

err_free_context:
	poptFreeContext(cmd_context);
}


void err_cleanup(GError **errp)
{
	g_return_if_fail(errp!=NULL);

	if (!*errp)
		return;
	printf("\nERROR: %s\n",(*errp)->message);
	g_clear_error(errp);
}


void invoke_cmd(int cmd_argc,const char **cmd_argv)
{
GError *gerr=NULL;

	invoke_cmd_err(cmd_argc,cmd_argv,&gerr);
	err_cleanup(&gerr);
}


/* Returns: Success (no error occured). */
gboolean errvfsresult_to_gerr(GError **errp,GnomeVFSResult errvfsresult)
{
	g_return_val_if_fail(!errp || !*errp,FALSE);

	if (errvfsresult==GNOME_VFS_OK)
		return TRUE;

	g_set_error(errp,CMDLINE_MAIN_ERROR,CMDLINE_MAIN_ERROR_GENERIC_ERROR,
			_("Generic error: %s"),gnome_vfs_result_to_string(errvfsresult));
	return FALSE;
}

void main_exit(void) G_GNUC_NORETURN;
void main_exit(void)
{
	if (cmdline_captive_vfs_object) {
		g_object_unref(cmdline_captive_vfs_object);
		cmdline_captive_vfs_object=NULL;
		}
	exit(EXIT_SUCCESS);
}

int main(int argc,char **argv)
{
poptContext context;
int errint;
const char **cmd_argv,**csp;
int cmd_argc;
GError *gerr=NULL;
struct captive_options options;

	/* Do not set g_log_set_always_fatal() here as we would not be able
	 * to restart failed children due to communication-failure alarms.
	 */

	captive_standalone_init();

	captive_options_init(&options);
	captive_options=&options;	/* for parsing by 'CAPTIVE_POPT_INCLUDE' */

	context=poptGetContext(
			PACKAGE,	/* name */
			argc,(/*en-const*/const char **)argv,	/* argc,argv */
			popt_table,	/* options */
			POPT_CONTEXT_POSIXMEHARDER);	/* flags; && !POPT_CONTEXT_KEEP_FIRST */
	if (context==NULL) {
		g_error(_("Error parsing command-line arguments"));
		return EXIT_FAILURE;
		}
	errint=poptReadDefaultConfig(context,
			TRUE);	/* useEnv */
	if (errint!=0)
		g_warning(_("Error reading default popt configuration"));
	errint=poptGetNextOpt(context);
	if (errint!=-1) {
		g_error(_("Error parsing (dash-prefixed) command-line argument"));
		return EXIT_FAILURE;
		}
	cmd_argv=poptGetArgs(context);
	for (csp=cmd_argv,cmd_argc=0;csp && *csp;csp++)
		cmd_argc++;

	captive_options=NULL;	/* already parsed by 'CAPTIVE_POPT_INCLUDE' */

	/* image_iochannel */
	if (cmd_argc<=0) {
		g_error(_("File/device disk image pathname command-line argument required"));
		return EXIT_FAILURE;
		}
	g_assert(options.image_iochannel==NULL);
	if (!(options.image_iochannel=g_io_channel_new_file(
			cmd_argv[0],	/* filename */
			(options.rwmode==CAPTIVE_OPTION_RWMODE_RW ? "r+" : "r"),	/* mode */
			NULL))) {	/* error */
		g_error(_("image_iochannel open failed"));
		return EXIT_FAILURE;
		}
	cmd_argc--;
	cmd_argv++;

	if (options.filesystem.type==CAPTIVE_OPTIONS_MODULE_TYPE_EMPTY) {
		g_error(_("'--filesystem' option required ('ntfs.sys' pathname suggested)"));
		return EXIT_FAILURE;
		}
	if (!options.load_module) {
		g_warning(_("'--load-module' option required ('ntoskrnl.exe' pathname suggested)"));
		return EXIT_FAILURE;
		}

	if (GNOME_VFS_OK!=captive_vfs_new(&cmdline_captive_vfs_object,&options)) {
		g_error(_("captive_vfs_new() failed"));
		return EXIT_FAILURE;
		}
	captive_options_free(&options);

	cmd_cd_internal("/",&gerr);
	if (gerr) {
		err_cleanup(&gerr);
		return EXIT_FAILURE;
		}

	invoke_cmd(cmd_argc,cmd_argv);

	/* 'cmd_argv' gets cleared by 'poptFreeContext(context);' below */
	poptFreeContext(context);

	main_exit();	/* unref 'cmdline_captive_vfs_object' */
	/* NOTREACHED */
}
