/* $Id: capfuse_captive_file_info_object.c,v 1.1 2006/01/01 07:24:34 lace Exp $
 * Client fuse interface handling of 'GnomeVFSFileInfo' for libcaptive
 * Copyright (C) 2005 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include <glib/gmessages.h>
#include <errno.h>
#include <sys/stat.h>
#include <time.h>
#include <sys/time.h>

#include <captive/client-file-info.h>
#include <captive/client-vfs.h>

#include "capfuse_captive_file_info_object.h"	/* self */
#include "gnomevfsresult.h"
#include "main.h"


static CaptiveVfsVolumeInfo volume_info;
static gboolean volume_info_valid=FALSE;


int capfuse_captive_file_info_object_to_stat(struct stat *stat,CaptiveFileInfoObject *captive_file_info_object)
{
guint nsec;

	g_return_val_if_fail(stat!=NULL,-EINVAL);
	g_return_val_if_fail(CAPTIVE_FILE_INFO_IS_OBJECT(captive_file_info_object),-EINVAL);

	if (!volume_info_valid) {
GnomeVFSResult errvfsresult;

		if (GNOME_VFS_OK!=(errvfsresult=captive_vfs_volume_info_get(capfuse_captive_vfs_object,&volume_info)))
			return -gnomevfsresult_to_errno(errvfsresult);
		volume_info_valid=TRUE;
		}

	stat->st_mode=0;

	if (captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_TYPE)
		switch (captive_file_info_object->p.type) {
			case GNOME_VFS_FILE_TYPE_REGULAR:
				stat->st_mode|=S_IFREG;
				if (1 /* !private */)
					stat->st_mode|=0444;
				break;
			case GNOME_VFS_FILE_TYPE_DIRECTORY:
				stat->st_mode|=S_IFDIR;
				if (1 /* !private */)
					stat->st_mode|=0555;
				break;
			default:
				g_warning("Unknown GnomeVFSFileInfo.type=%d of: %s",(int)captive_file_info_object->p.type,captive_file_info_object->p.name);
				return -EINVAL;
			}
	if (captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS)
		stat->st_mode|=captive_file_info_object->p.permissions & 0777;
	else
		stat->st_mode|=0600;

	if (!(captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_LINK_COUNT))
		stat->st_nlink=1;
	else
		stat->st_nlink=captive_file_info_object->p.link_count;

	stat->st_uid=0;	/* we own the file */
	stat->st_gid=0;	/* we own the file */

	if (!(captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_SIZE))
		stat->st_size=0;
	else
		stat->st_size=captive_file_info_object->p.size;

	if (!(captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_ATIME))
		capfuse_timespec_split_get(&stat->st_atime,&nsec);
	else {
		stat->st_atime=captive_file_info_object->p.atime;
		nsec=captive_file_info_object->atime_nsec;
		}
#if defined(HAVE_STRUCT_STAT_ST_ATIM)
	g_assert(&stat->st_atim.tv_sec==&stat->st_atime);
	stat->st_atim.tv_nsec=nsec;
#elif defined(HAVE_STRUCT_STAT_ST_ATIMENSEC)
	stat->st_atimensec=nsec;
#endif

	if (!(captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_MTIME))
		capfuse_timespec_split_get(&stat->st_mtime,&nsec);
	else {
		stat->st_mtime=captive_file_info_object->p.mtime;
		nsec=captive_file_info_object->mtime_nsec;
		}
#if defined(HAVE_STRUCT_STAT_ST_MTIM)
	g_assert(&stat->st_mtim.tv_sec==&stat->st_mtime);
	stat->st_mtim.tv_nsec=nsec;
#elif defined(HAVE_STRUCT_STAT_ST_MTIMENSEC)
	stat->st_mtimensec=nsec;
#endif

	if (!(captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_CTIME))
		capfuse_timespec_split_get(&stat->st_ctime,&nsec);
	else {
		stat->st_ctime=captive_file_info_object->p.ctime;
		nsec=captive_file_info_object->ctime_nsec;
		}
#if defined(HAVE_STRUCT_STAT_ST_CTIM)
	g_assert(&stat->st_ctim.tv_sec==&stat->st_ctime);
	stat->st_ctim.tv_nsec=nsec;
#elif defined(HAVE_STRUCT_STAT_ST_CTIMENSEC)
	stat->st_ctimensec=nsec;
#endif

	stat->st_blksize=volume_info.block_size;

	if (!(captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_BLOCK_COUNT)) {
		/* Rounding? */
		/* 0? */
		stat->st_blocks=stat->st_size/volume_info.block_size;
		}
	else
		stat->st_blocks=captive_file_info_object->p.block_count;

	stat->st_dev=0;
	stat->st_rdev=0;	/* Never to be used. */
	stat->st_ino=0;

	return 0;
}


void capfuse_timespec_split_get(time_t *sec_return,guint *nsec_return)
{
int errint;
struct timeval timeval;

	g_return_if_fail(sec_return!=NULL);
	g_return_if_fail(nsec_return!=NULL);

	errint=gettimeofday(
			&timeval,	/* (struct timeval *)tv */
			NULL);	/* (struct timezone *)tz */
	g_assert(!errint);

	*sec_return=timeval.tv_sec;
	/* FIXME: Where to get 'usec' precision? */
	*nsec_return=timeval.tv_usec*1000;
}
