/* $Id: giognomevfs.c,v 1.11 2003/12/06 14:04:54 short Exp $
 * glib GIOChannel interface over gnome-vfs GnomeVFSURI for libcaptive
 * Copyright (C) 2002-2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "captive/macros.h"
#include <libgnomevfs/gnome-vfs-method.h>
#include <libgnomevfs/gnome-vfs-ops.h>
#include <glib/gmessages.h>
#include "gnome-vfs-module.h"
#include "captive/options.h"
#include "../../libcaptive/client/lib.h"	/* for captive_giochannel_setup(); FIXME: pathname */


/* FIXME: fill 'err' */

struct captive_gnomevfs_giognomevfs {
	GIOChannel iochannel;
	GnomeVFSHandle *gnomevfshandle;
	};

G_LOCK_DEFINE_STATIC(giochannel_funcs);
static GIOFuncs giochannel_funcs;


static gboolean validate_giognomevfs(struct captive_gnomevfs_giognomevfs *giognomevfs)
{
	g_return_val_if_fail(giognomevfs!=NULL,FALSE);
	g_return_val_if_fail(giognomevfs->gnomevfshandle!=NULL,FALSE);

	return TRUE;
}


static GIOStatus captive_gnomevfs_giognomevfs_io_read
		(GIOChannel *channel,gchar *buf,gsize count,gsize *bytes_read,GError **err)
{
struct captive_gnomevfs_giognomevfs *giognomevfs=(struct captive_gnomevfs_giognomevfs *)channel;
GnomeVFSResult errvfsresult;
GnomeVFSFileSize bytes_read_local;

	g_return_val_if_fail(validate_giognomevfs(giognomevfs),G_IO_STATUS_ERROR);
	g_return_val_if_fail(buf!=NULL,G_IO_STATUS_ERROR);
	g_return_val_if_fail(bytes_read!=NULL,G_IO_STATUS_ERROR);

	errvfsresult=gnome_vfs_read(
			giognomevfs->gnomevfshandle,	/* handle */
			buf,	/* buffer */
			count,	/* bytes */
			&bytes_read_local);	/* bytes_read */
	/* During seek in block device such as on URL file:///dev/hda1#captive-fastfat.sys-ro:/
	 * we will do llseek(2) on "/dev/hda1" device from captive_giochannel_size().
	 * During read on the end boundary of Linux kernel block device we will
	 * get GNOME_VFS_ERROR_IO at least from linux-kernel-2.4.19-ac4
	 * therefore it must be accepted without complaints by us.
	 * FIXME: It would be nice to detect the state when we are called from captive_giochannel_size().
	 */
	if (errvfsresult==GNOME_VFS_ERROR_IO) {
		*bytes_read=0;
		return G_IO_STATUS_ERROR;
		}
	g_return_val_if_fail((errvfsresult==GNOME_VFS_OK || errvfsresult==GNOME_VFS_ERROR_EOF),G_IO_STATUS_ERROR);

	*bytes_read=bytes_read_local;
	return (errvfsresult==GNOME_VFS_ERROR_EOF ? G_IO_STATUS_EOF : G_IO_STATUS_NORMAL);
}


static GIOStatus captive_gnomevfs_giognomevfs_io_write
		(GIOChannel *channel,const gchar *buf,gsize count,gsize *bytes_written,GError **err)
{
struct captive_gnomevfs_giognomevfs *giognomevfs=(struct captive_gnomevfs_giognomevfs *)channel;
GnomeVFSResult errvfsresult;
GnomeVFSFileSize bytes_written_local;

	g_return_val_if_fail(validate_giognomevfs(giognomevfs),G_IO_STATUS_ERROR);
	g_return_val_if_fail(buf!=NULL,G_IO_STATUS_ERROR);
	g_return_val_if_fail(bytes_written!=NULL,G_IO_STATUS_ERROR);

	errvfsresult=gnome_vfs_write(
			giognomevfs->gnomevfshandle,	/* handle */
			buf,	/* buffer */
			count,	/* bytes */
			&bytes_written_local);	/* bytes_written */
	g_return_val_if_fail(errvfsresult==GNOME_VFS_OK,G_IO_STATUS_ERROR);

	*bytes_written=bytes_written_local;
	return G_IO_STATUS_NORMAL;
}


static GIOStatus captive_gnomevfs_giognomevfs_io_seek(GIOChannel *channel,gint64 offset,GSeekType type,GError **err)
{
struct captive_gnomevfs_giognomevfs *giognomevfs=(struct captive_gnomevfs_giognomevfs *)channel;
GnomeVFSSeekPosition whence;
GnomeVFSResult errvfsresult;

	g_return_val_if_fail(validate_giognomevfs(giognomevfs),G_IO_STATUS_ERROR);

	switch (type) {
		case G_SEEK_CUR: whence=GNOME_VFS_SEEK_CURRENT; break;
		case G_SEEK_SET: whence=GNOME_VFS_SEEK_START;   break;
		case G_SEEK_END: whence=GNOME_VFS_SEEK_END;     break;
		default: g_return_val_if_reached(G_IO_STATUS_ERROR);
		}
	errvfsresult=gnome_vfs_seek(giognomevfs->gnomevfshandle,whence,offset);
	/* During seek in block device such as on URL file:///dev/hda1#captive-fastfat.sys-ro:/
	 * we will do llseek(2) on "/dev/hda1" device from captive_giochannel_size().
	 * Although we are allowed to seek behind EOF on regular files
	 * at least linux-kernel-2.4.19-ac4/fs/block_dev.c/block_llseek() will give
	 * EINVAL on seek behind EOF therefore it must be accepted without complaints by us.
	 */
	if (errvfsresult!=GNOME_VFS_OK)
		return G_IO_STATUS_ERROR;

	return G_IO_STATUS_NORMAL;
}


static GIOStatus captive_gnomevfs_giognomevfs_io_close(GIOChannel *channel,GError **err)
{
struct captive_gnomevfs_giognomevfs *giognomevfs=(struct captive_gnomevfs_giognomevfs *)channel;

	g_return_val_if_fail(validate_giognomevfs(giognomevfs),G_IO_STATUS_ERROR);

	gnome_vfs_close(giognomevfs->gnomevfshandle);

	return G_IO_STATUS_NORMAL;
}


static GSource* captive_gnomevfs_giognomevfs_io_create_watch(GIOChannel *channel,GIOCondition condition)
{
struct captive_gnomevfs_giognomevfs *giognomevfs=(struct captive_gnomevfs_giognomevfs *)channel;

	g_return_val_if_fail(validate_giognomevfs(giognomevfs),NULL);

	g_return_val_if_reached(NULL);	/* FIXME: NOT IMPLEMENTED YET */
}


static void captive_gnomevfs_giognomevfs_io_free(GIOChannel *channel)
{
struct captive_gnomevfs_giognomevfs *giognomevfs=(struct captive_gnomevfs_giognomevfs *)channel;

	g_return_if_fail(validate_giognomevfs(giognomevfs));

	g_free(giognomevfs);
}


static GIOStatus captive_gnomevfs_giognomevfs_io_set_flags(GIOChannel *channel,GIOFlags flags,GError **err)
{
struct captive_gnomevfs_giognomevfs *giognomevfs=(struct captive_gnomevfs_giognomevfs *)channel;

	g_return_val_if_fail(validate_giognomevfs(giognomevfs),G_IO_STATUS_ERROR);

	/* no G_IO_FLAG_APPEND and no G_IO_FLAG_NONBLOCK */
	g_return_val_if_fail((flags&G_IO_FLAG_SET_MASK)==0,G_IO_STATUS_ERROR);

	return G_IO_STATUS_NORMAL;
}


static GIOFlags captive_gnomevfs_giognomevfs_io_get_flags(GIOChannel *channel)
{
struct captive_gnomevfs_giognomevfs *giognomevfs=(struct captive_gnomevfs_giognomevfs *)channel;

	g_return_val_if_fail(validate_giognomevfs(giognomevfs),0);

	return 0;	/* | !G_IO_FLAG_APPEND | !G_IO_FLAG_NONBLOCK */
}


GnomeVFSResult captive_gnomevfs_giognomevfs_new
		(struct captive_gnomevfs_giognomevfs **giognomevfsp,GnomeVFSURI *uri,enum captive_option_rwmode rwmode)
{
GnomeVFSHandle *handle;
struct captive_gnomevfs_giognomevfs *giognomevfs;
GnomeVFSResult errvfsresult;

	g_return_val_if_fail(giognomevfsp!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(uri!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	errvfsresult=gnome_vfs_open_uri(&handle,uri,
			0	/* open_mode */
					| GNOME_VFS_OPEN_RANDOM	/* mandatory; otherwise we will get truncation if GNOME_VFS_OPEN_WRITE */
					| GNOME_VFS_OPEN_READ
					| (rwmode==CAPTIVE_OPTION_RWMODE_RW ? GNOME_VFS_OPEN_WRITE : 0));
	g_return_val_if_fail(errvfsresult==GNOME_VFS_OK,errvfsresult);

	G_LOCK(giochannel_funcs);
	giochannel_funcs.io_read        =captive_gnomevfs_giognomevfs_io_read;
	giochannel_funcs.io_write       =captive_gnomevfs_giognomevfs_io_write;
	giochannel_funcs.io_seek        =captive_gnomevfs_giognomevfs_io_seek;
	giochannel_funcs.io_close       =captive_gnomevfs_giognomevfs_io_close;
	giochannel_funcs.io_create_watch=captive_gnomevfs_giognomevfs_io_create_watch;
	giochannel_funcs.io_free        =captive_gnomevfs_giognomevfs_io_free;
	giochannel_funcs.io_set_flags   =captive_gnomevfs_giognomevfs_io_set_flags;
	giochannel_funcs.io_get_flags   =captive_gnomevfs_giognomevfs_io_get_flags;
	G_UNLOCK(giochannel_funcs);

	captive_new(giognomevfs);
	g_assert(G_STRUCT_OFFSET(struct captive_gnomevfs_giognomevfs,iochannel)==0);	/* safely re-type-able */
	g_io_channel_init(&giognomevfs->iochannel);
	giognomevfs->iochannel.funcs=&giochannel_funcs;
	giognomevfs->iochannel.is_seekable=TRUE;
	giognomevfs->iochannel.is_readable=TRUE;
	giognomevfs->iochannel.is_writeable=(rwmode!=CAPTIVE_OPTION_RWMODE_RO);
	giognomevfs->iochannel.close_on_unref=TRUE;	/* run g_io_channel_shutdown() flush on last unref */
	giognomevfs->gnomevfshandle=handle;

	captive_giochannel_setup(&giognomevfs->iochannel);

	*giognomevfsp=giognomevfs;
	return GNOME_VFS_OK;
}
