/* $Id: captivefs-attr.c,v 1.3 2003/11/29 18:35:12 short Exp $
 * lufs interface module misc functions for libcaptive
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "captivefs-attr.h"	/* self */
#include <glib/gmessages.h>
#include <time.h>
#include "captivefs-vfs.h"


gboolean captivefs_GnomeVFSFileInfo_to_lufs_fattr
		(struct captivefs_vfs *captivefs_vfs,struct lufs_fattr *fattr,const GnomeVFSFileInfo *file_info)
{
	g_return_val_if_fail(captivefs_vfs_validate(captivefs_vfs),FALSE);
	g_return_val_if_fail(fattr!=NULL,FALSE);
	g_return_val_if_fail(file_info!=NULL,FALSE);

	fattr->f_mode=0;

	if (file_info->valid_fields&GNOME_VFS_FILE_INFO_FIELDS_TYPE)
		switch (file_info->type) {
			case GNOME_VFS_FILE_TYPE_REGULAR:
				fattr->f_mode|=S_IFREG;
				/* 'fmode' kernel LUFS option only |= to our reported 'f_mode': */
				if (!captivefs_vfs->private)
					fattr->f_mode|=0444;
				break;
			case GNOME_VFS_FILE_TYPE_DIRECTORY:
				fattr->f_mode|=S_IFDIR;
				/* 'dmode' kernel LUFS option only |= to our reported 'f_mode': */
				if (!captivefs_vfs->private)
					fattr->f_mode|=0555;
				break;
			default:
				g_warning("Unknown GnomeVFSFileInfo.type=%d of: %s",(int)file_info->type,file_info->name);
				return FALSE;
				}
	if (file_info->valid_fields&GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS)
		fattr->f_mode|=file_info->permissions & 0777;
	else
		fattr->f_mode|=0600;

	if (!(file_info->valid_fields&GNOME_VFS_FILE_INFO_FIELDS_LINK_COUNT))
		fattr->f_nlink=1;
	else
		fattr->f_nlink=file_info->link_count;

	fattr->f_uid=1;	/* we own the file */
	fattr->f_gid=1;	/* we own the file */

	if (!(file_info->valid_fields&GNOME_VFS_FILE_INFO_FIELDS_SIZE))
		fattr->f_size=0;
	else
		fattr->f_size=file_info->size;

	if (!(file_info->valid_fields&GNOME_VFS_FILE_INFO_FIELDS_ATIME))
		fattr->f_atime=time(NULL);
	else
		fattr->f_atime=file_info->atime;

	if (!(file_info->valid_fields&GNOME_VFS_FILE_INFO_FIELDS_MTIME))
		fattr->f_mtime=time(NULL);
	else
		fattr->f_mtime=file_info->mtime;

	if (!(file_info->valid_fields&GNOME_VFS_FILE_INFO_FIELDS_CTIME))
		fattr->f_ctime=time(NULL);
	else
		fattr->f_ctime=file_info->ctime;

	return TRUE;
}


gboolean captivefs_lufs_fattr_to_GnomeVFSFileInfo(GnomeVFSFileInfo *file_info,const struct lufs_fattr *fattr)
{
	g_return_val_if_fail(file_info!=NULL,FALSE);
	g_return_val_if_fail(fattr!=NULL,FALSE);

	file_info->valid_fields=0;

	switch (fattr->f_mode&S_IFMT) {
		case S_IFREG: file_info->type=GNOME_VFS_FILE_TYPE_REGULAR;   break;
		case S_IFDIR: file_info->type=GNOME_VFS_FILE_TYPE_DIRECTORY; break;
		default:
			g_warning("lufs_fattr_to_GnomeVFSFileInfo: f_mode&S_IFMT=0x%lX",(long)(fattr->f_mode&S_IFMT));
			return FALSE;
		}
	file_info->valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_TYPE;

	file_info->permissions=fattr->f_mode & 0777;
	file_info->valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS;

	file_info->link_count=fattr->f_nlink;
	file_info->valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_LINK_COUNT;

	/* ignore: fattr->f_uid; we own the file */
	/* ignore: fattr->f_gid; we own the file */

	file_info->size=fattr->f_size;
	file_info->valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_SIZE;

	if ((file_info->atime=fattr->f_atime))
		file_info->valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_ATIME;

	if ((file_info->mtime=fattr->f_mtime))
		file_info->valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_MTIME;

	if ((file_info->ctime=fattr->f_ctime))
		file_info->valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_CTIME;

	return TRUE;
}
