/* $Id: captivefs-directory.c,v 1.6 2003/12/13 21:20:05 short Exp $
 * lufs interface module directory objects implementation for libcaptive
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include <glib/gmessages.h>
#include "captivefs-misc.h"
#include "captivefs-attr.h"
#include "captivefs-vfs.h"

#include <captive/client-vfs.h>
#include <captive/client-directory.h>

#include <lufs/fs.h>
#include <lufs/proto.h>


/* Read a directory's content
 * For each directory entry, call 
 *    lu_cache_add2dir(struct directory *dir, char *name, char *link, struct lufs_fattr *fattr)
 * to add its information.
 * The link is optional(NULL) and applicable only if the entry is a 
 * (sym)link and we have the target info at hand.
 *
 * Notes:
 *     dir_name is an absolute path.  However, it is generally a good idea
 * to either change to that directory here or at least keep track of what
 * directory was being called on, for function calls that might not get
 * absolute paths.
 *     If your filesystem doesn't natively support '.' or '..', don't forget
 * to add them to the cache first-thing here.
 */
int captivefs_readdir(struct captivefs_vfs *captivefs_vfs,const char *dir_name,struct directory *ddir)
{
struct lufs_fattr fattr;
GnomeVFSResult errvfsresult;
CaptiveDirectoryObject *captive_directory_object;
GnomeVFSFileInfo file_info;
const char *dots[]={".","..",NULL},**csp;

	g_return_val_if_fail(captivefs_vfs_validate(captivefs_vfs),-1);
	g_return_val_if_fail(dir_name!=NULL,-1);
	g_return_val_if_fail(ddir!=NULL,-1);

	if (captivefs_vfs->options.debug_messages)
		g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_readdir: dir_name=%s",dir_name);

	dir_name=CAPTIVEFS_FILENAME_TO_UTF8_ALLOCA(dir_name);

	G_LOCK(libcaptive);
	errvfsresult=captive_directory_new_open(&captive_directory_object,captivefs_vfs->captive_vfs_object,dir_name);
	G_UNLOCK(libcaptive);
	if (errvfsresult!=GNOME_VFS_OK)
		goto fail;

	file_info.valid_fields=GNOME_VFS_FILE_INFO_FIELDS_TYPE;
	file_info.type=GNOME_VFS_FILE_TYPE_DIRECTORY;
	if (!captivefs_GnomeVFSFileInfo_to_lufs_fattr(captivefs_vfs,&fattr,&file_info))
		goto fail_unref;
	for (csp=dots;*csp;csp++)
		if (0>lu_cache_add2dir(ddir,(/* de-const */ char *)*csp,NULL,&fattr)) {
			g_warning("Failed lu_cache_add2dir() for: %s",*csp);
			goto fail_unref;
			}

	for (;;) {
char *file_info_name_filename;
int errint;

		G_LOCK(libcaptive);
		errvfsresult=captive_directory_read(captive_directory_object,&file_info);
		G_UNLOCK(libcaptive);
		if (errvfsresult==GNOME_VFS_ERROR_EOF) {
			/* 'captive_directory_object' is now stuck at EOF - GnomeVFS behaves that way. */
			break;
			}
		if (errvfsresult!=GNOME_VFS_OK)
			goto fail_unref;

		if (!captivefs_GnomeVFSFileInfo_to_lufs_fattr(captivefs_vfs,&fattr,&file_info))
			goto fail_unref;

		file_info_name_filename=captivefs_filename_from_utf8_malloc_errorchecking(file_info.name);
		errint=lu_cache_add2dir(ddir,
				(file_info_name_filename ? file_info_name_filename : file_info.name),
				NULL,&fattr);
		g_free(file_info_name_filename);	/* may be NULL */
		if (0>errint) {
			g_warning("Failed lu_cache_add2dir() for: %s",file_info.name);
			goto fail_unref;
			}

		if (captivefs_vfs->options.debug_messages)
			g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_readdir: got: %s",file_info.name);
		}

	G_LOCK(libcaptive);
	g_object_unref(captive_directory_object);
	G_UNLOCK(libcaptive);

	return 0;

fail_unref:
	G_LOCK(libcaptive);
	g_object_unref(captive_directory_object);
	G_UNLOCK(libcaptive);
fail:
	return -1;
}


/* Create a directory
 */
int captivefs_mkdir(struct captivefs_vfs *captivefs_vfs,const char *dir,int mode)
{
CaptiveDirectoryObject *captive_directory_object;
GnomeVFSResult errvfsresult;

	g_return_val_if_fail(captivefs_vfs_validate(captivefs_vfs),-1);
	g_return_val_if_fail(dir!=NULL,-1);

	if (captivefs_vfs->options.debug_messages)
		g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_mkdir: dir=%s,mode=0x%X",dir,mode);

	dir=CAPTIVEFS_FILENAME_TO_UTF8_ALLOCA(dir);

	G_LOCK(libcaptive);
	errvfsresult=captive_directory_new_make(&captive_directory_object,captivefs_vfs->captive_vfs_object,dir,mode);
	G_UNLOCK(libcaptive);
	if (errvfsresult!=GNOME_VFS_OK)
		return -1;

	G_LOCK(libcaptive);
	g_object_unref(captive_directory_object);
	G_UNLOCK(libcaptive);

	return 0;
}


/* Delete a directory
 */
int captivefs_rmdir(struct captivefs_vfs *captivefs_vfs,const char *dir)
{
CaptiveDirectoryObject *captive_directory_object;
GnomeVFSResult errvfsresult;

	g_return_val_if_fail(captivefs_vfs_validate(captivefs_vfs),-1);
	g_return_val_if_fail(dir!=NULL,-1);

	if (captivefs_vfs->options.debug_messages)
		g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_rmdir: dir=%s",dir);

	dir=CAPTIVEFS_FILENAME_TO_UTF8_ALLOCA(dir);

	G_LOCK(libcaptive);
	errvfsresult=captive_directory_new_open(&captive_directory_object,captivefs_vfs->captive_vfs_object,dir);
	G_UNLOCK(libcaptive);
	if (errvfsresult!=GNOME_VFS_OK)
		return -1;

	G_LOCK(libcaptive);
	errvfsresult=captive_directory_remove(captive_directory_object);
	G_UNLOCK(libcaptive);

	G_LOCK(libcaptive);
	g_object_unref(captive_directory_object);
	G_UNLOCK(libcaptive);

	if (errvfsresult!=GNOME_VFS_OK)
		return -1;
	return 0;
}
