/* $Id: captivefs-vfs.c,v 1.12 2005/12/22 08:22:56 lace Exp $
 * lufs interface module vfs objects implementation for libcaptive
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "captivefs-vfs.h"	/* self */
#include <glib/gmessages.h>
#include "captivefs-misc.h"
#include <unistd.h>
#include <locale.h>

#include <captive/client-vfs.h>
#include <captive/options.h>
#include <captive/macros.h>
#include <captive/client.h>

#include <lufs/fs.h>
#include <lufs/proto.h>


gboolean captivefs_vfs_validate(struct captivefs_vfs *captivefs_vfs)
{
	g_return_val_if_fail(captivefs_vfs!=NULL,FALSE);

	G_LOCK(libcaptive);
	g_assert(captivefs_vfs->inits>0);
	g_assert(captivefs_vfs->mounts>=0);
	g_assert(captivefs_vfs->mounts<=captivefs_vfs->inits);
	if (!captivefs_vfs->captive_vfs_object) {
GnomeVFSResult errvfsresult;

		if (captivefs_vfs->parent_pid==getpid()) {
			G_UNLOCK(libcaptive);
			return FALSE;
			}

		errvfsresult=captive_vfs_new(&captivefs_vfs->captive_vfs_object,&captivefs_vfs->options);

		if (errvfsresult!=GNOME_VFS_OK) {
			G_UNLOCK(libcaptive);
			g_return_val_if_reached(FALSE);
			}
		}
	if (!CAPTIVE_VFS_IS_OBJECT(captivefs_vfs->captive_vfs_object)) {
		G_UNLOCK(libcaptive);
		g_return_val_if_reached(FALSE);
		}
	G_UNLOCK(libcaptive);

	return TRUE;
}


/* Initialization
 * Here we allocate a structure to hold all the file system local info 
 * (localfs_local). This structure will then be passed as a parameter to 
 * the other functions.
 * global_ctx holds info about another structure that can be shared between all
 * instances of the filesystem. If the pointer is NULL, then this is the
 * first instance and the structure should be allocated.
 * ! You must implement  locking/usage-count/deallocation logic when using
 *   a global context. (locking is omited here)
 * ! Most filesystems don't need a global context so you can safely ignore the
 *   global_ctx parameter.  
 */
struct captivefs_vfs *captivefs_init
		(struct list_head *cfg,struct dir_cache *cache,const struct credentials *cred,struct captivefs_vfs **global_ctx)
{
struct captivefs_vfs *captivefs_vfs;
const gchar *cgs;
gchar *gs,*captive_options_string;

	g_return_val_if_fail(global_ctx!=NULL,NULL);

	/* Do not: g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_init");
	 * as we do not yet have 'VFS_DEBUG_MESSAGES(captive_vfs_object)' set.
	 * Generally we make all g_log() conditional here as we do not want to mess
	 * with overriden GLog handlers of libcaptive itself.
	 */

	G_LOCK(libcaptive);
	if ((captivefs_vfs=*global_ctx)) {
		g_assert(captivefs_vfs->inits>0);
		/* We do not support multiple LUFS threads if they could not be cross-locked. */
		g_return_val_if_fail(g_thread_supported(),NULL);
		captivefs_vfs->inits++;
		G_UNLOCK(libcaptive);
		if (captivefs_vfs->options.debug_messages)
			g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_init");
		return captivefs_vfs;
		}
	G_UNLOCK(libcaptive);

	captive_standalone_init();

	if (!g_thread_supported())
		g_thread_init(NULL);	/* g_thread_init() fails on second initialization */

	captive_new(captivefs_vfs);
	captivefs_vfs->captive_vfs_object=NULL;
	captivefs_vfs->private=FALSE;
	captivefs_vfs->parent_pid=getpid();
	captivefs_vfs->global_ctx=global_ctx;
	*global_ctx=captivefs_vfs;
	captivefs_vfs->inits=1;
	captivefs_vfs->mounts=0;

	captive_options_init(&captivefs_vfs->options);

	if ((cgs=lu_opt_getchar(cfg,"MOUNT","captive_options"))) {
		captive_options_string=(/* de-const */gchar *)captive_strdup_alloca(cgs);
		/* Convert ';' to ' ' to prevent its parsing by LUFS to keep its ordering
		 * to let the options be overridable by user (such as 'ro').
		 */
		for (gs=captive_options_string;(gs=strchr(gs,';'));gs++)
			*gs=' ';
		if (!captive_options_parse(
				&captivefs_vfs->options,	/* options */
				captive_options_string))	/* captive_args */
			goto fail_free_options;
		}
	if (lu_opt_getchar(cfg,"MOUNT","private"))
		captivefs_vfs->private=TRUE;

	if (captivefs_vfs->options.debug_messages)
		g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_init");

	/* image_iochannel */
	if ((cgs=lu_opt_getchar(cfg,"MOUNT","image"))) {
		g_assert(captivefs_vfs->options.image_iochannel==NULL);
		if (!(captivefs_vfs->options.image_iochannel=g_io_channel_new_file(
				cgs,	/* filename */
				(captivefs_vfs->options.rwmode==CAPTIVE_OPTION_RWMODE_RW ? "r+" : "r"),	/* mode */
				NULL))) {	/* error */
			g_warning(_("%s: image_iochannel open failed"),"captivefs_init");
			goto fail_free_options;
			}
		}

	return captivefs_vfs;

fail_free_options:
	captive_options_free(&captivefs_vfs->options);
/* fail: */
	return NULL;
}


/* Cleanup
 * Check the global context count and free it if necessary.
 * Deallocate memory and free all resources.
 */
void captivefs_free(struct captivefs_vfs *captivefs_vfs)
{
	g_return_if_fail(captivefs_vfs_validate(captivefs_vfs));
	g_return_if_fail(captivefs_vfs->inits>0);

	if (captivefs_vfs->options.debug_messages)
		g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_free");
	
	if (--captivefs_vfs->inits>0) {
		g_assert(captivefs_vfs_validate(captivefs_vfs));
		return;
		}
	g_assert(captivefs_vfs->mounts==0);
	G_LOCK(libcaptive);
	*captivefs_vfs->global_ctx=NULL;
	G_UNLOCK(libcaptive);

	g_assert(G_OBJECT(captivefs_vfs->captive_vfs_object)->ref_count==1);

	G_LOCK(libcaptive);
	g_object_unref(captivefs_vfs->captive_vfs_object);
	G_UNLOCK(libcaptive);

	/* Do not: g_io_channel_unref(captivefs_vfs->options.image_iochannel);
	 * as it will be unreferenced by captive_options_free().
	 */
	captive_options_free(&captivefs_vfs->options);
	g_free(captivefs_vfs);
}


/* Mount the file system.
 * Called when a mount operation is performed.
 * Initialize specific connections, login, etc.
 *
 * Notes:
 *     By default, LUFS may attempt multiple connections at once.  If your
 * filesystem doesn't support this, you need to specificy -c 1 on the
 * lufsmount command line or connections=1 in the mount options.
 *     See ftpfs for an example of how to read configuration options
 * from a configuration file if you want to, for example, be able to set
 * default values.
 */
int captivefs_mount(struct captivefs_vfs *captivefs_vfs)
{
	/* We may be called from the parent. */
	g_return_val_if_fail(captivefs_vfs!=NULL,FALSE);
	captivefs_vfs_validate(captivefs_vfs);	/* It may return FALSE. */

	if (captivefs_vfs->options.debug_messages)
		g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_mount");

	captivefs_vfs->mounts++;

	captivefs_vfs_validate(captivefs_vfs);	/* It may return FALSE. */

	return 1;	/* NEVER return 0 */
}


/* Unmount the  file system
 * Called when the file system is unmounted.
 */
void captivefs_umount(struct captivefs_vfs *captivefs_vfs)
{
	g_return_if_fail(captivefs_vfs_validate(captivefs_vfs));
	g_return_if_fail(captivefs_vfs->mounts>0);

	if (captivefs_vfs->options.debug_messages)
		g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_umount");

	captivefs_vfs->mounts--;

	g_assert(captivefs_vfs_validate(captivefs_vfs));
}


#ifdef HAVE_STRUCT_LUFS_SBATTR
int captivefs_statfs(struct captivefs_vfs *captivefs_vfs,struct lufs_sbattr *sbattr)
{
CaptiveVfsVolumeInfo captive_volume_info;
GnomeVFSResult errvfsresult;

	g_return_val_if_fail(captivefs_vfs_validate(captivefs_vfs),-1);
	g_return_val_if_fail(sbattr!=NULL,-1);

	if (captivefs_vfs->options.debug_messages)
		g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"captivefs_statfs");

	G_LOCK(libcaptive);
	errvfsresult=captive_vfs_volume_info_get(captivefs_vfs->captive_vfs_object,&captive_volume_info);
	G_UNLOCK(libcaptive);
	if (errvfsresult!=GNOME_VFS_OK)
		return -1;

	sbattr->sb_bytes=captive_volume_info.bytes;
	sbattr->sb_bytes_free=captive_volume_info.bytes_free;
	sbattr->sb_bytes_available=captive_volume_info.bytes_available;
	/* 'sbattr->sb_files' not known - left unchanged. */
	/* 'sbattr->sb_ffree' not known - left unchanged. */

	return 0;
}
#endif /* HAVE_STRUCT_LUFS_SBATTR */
