/* $Id: cache.c,v 1.3 2003/12/07 06:32:34 short Exp $
 * reactos Cache Manager (Cc*) cache handling of libcaptive
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "privatebcbpin.h"
#include "reactos/ddk/status.h"


/**
 * CcPurgeCacheSection:
 * @SectionObjectPointer: Pointer specifying file to purge;
 * %NULL value is forbidden.
 * libcaptive interprets only #SharedCacheMap field as #PUBLIC_BCB pointer.
 * Field #SharedCacheMap value %NULL is permitted; libcaptive does a NOP with %TRUE return code in such case.
 * @FileOffset: Starting offset of the ranger to purge.
 * %NULL pointer is permitted and it means to purge the whole whole.
 * FIXME: Non %NULL pointer is NOT IMPLEMENTED YET by libcaptive.
 * @Length: Length of the range to purge. Ignored if @FileOffset==NULL.
 * @UninitializeCacheMaps: Purge also private cache maps (FIXME: ???).
 *
 * Drop any caching for shrunken file which is not being deleted.
 * libcaptive will no longer consider such #BCB as dirty.
 *
 * Undocumented: It is required during %FSCTL_LOCK_VOLUME by ntfs.sys of NT-5.1sp1
 * to return %TRUE value if #SharedCacheMap value is %NULL.
 *
 * Returns: %TRUE if the range was purged successfuly.
 */
BOOLEAN CcPurgeCacheSection(IN PSECTION_OBJECT_POINTERS SectionObjectPointer,
		IN PLARGE_INTEGER FileOffset OPTIONAL,IN ULONG Length,IN BOOLEAN UninitializeCacheMaps)
{
CaptiveSharedCacheMapObject *SharedCacheMap;
BOOLEAN r;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcPurgeCacheSection: SectionObjectPointer=0x%lX,->SharedCacheMap=0x%lX,FileOffset=0x%lX,Length=0x%lX,"
					"UninitializeCacheMaps=%d",
			(long)SectionObjectPointer,
			(!SectionObjectPointer ? -1 : (long)SectionObjectPointer->SharedCacheMap),
			(!FileOffset ? -1 : (long)FileOffset->QuadPart),Length,
					UninitializeCacheMaps);

	g_return_val_if_fail(SectionObjectPointer!=NULL,FALSE);
	g_return_val_if_fail(FileOffset==NULL,FALSE);	/* NOT IMPLEMENTED YET */
	g_return_val_if_fail(UninitializeCacheMaps==0,FALSE);	/* NOT IMPLEMENTED YET */

	if (SectionObjectPointer->SharedCacheMap) {
		SharedCacheMap=captive_SectionObjectPointers_to_SharedCacheMap(SectionObjectPointer);
		captive_shared_cache_map_purge(SharedCacheMap);
		}

	r=TRUE;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcPurgeCacheSection: r=%d",r);

	return r;
}


/**
 * CcFlushCache:
 * @SectionObjectPointer: Pointer specifying file to flush;
 * %NULL value is forbidden.
 * libcaptive interprets only #SharedCacheMap field as #PUBLIC_BCB pointer.
 * Field #SharedCacheMap value %NULL is permitted; libcaptive does a NOP in such case.
 * @FileOffset: Optional starting point of the range to flush.
 * %NULL value is permitted.
 * @Length: Length of the range to flush. Ignored if @FileOffset is %NULL.
 * @IoStatus: Optionally returns the resulting operation status.
 * #Information field will contain the number of bytes flushed.
 * %NULL value is permitted.
 *
 * Flushes out any pending dirty data in cache manager BCB mapping.
 * FIXME: libcaptive currently always flushes the full file ignoring any @FileOffset or @Length.
 *
 * VERIFIED: Ranged flushes.
 * VERIFIED: Synchronous write.
 */
VOID CcFlushCache(IN PSECTION_OBJECT_POINTERS SectionObjectPointer,
		IN PLARGE_INTEGER FileOffset OPTIONAL,IN ULONG Length,OUT PIO_STATUS_BLOCK IoStatus OPTIONAL)
{
CaptiveSharedCacheMapObject *SharedCacheMap;
guint64 flushed;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcFlushCache: SectionObjectPointer=0x%lX,->SharedCacheMap=0x%lX,FileOffset=0x%lX,Length=0x%lX",
			(long)SectionObjectPointer,
			(!SectionObjectPointer ? -1 : (long)SectionObjectPointer->SharedCacheMap),
			(!FileOffset ? -1 : (long)FileOffset->QuadPart),Length);

	g_return_if_fail(SectionObjectPointer!=NULL);

	if (SectionObjectPointer->SharedCacheMap) {
		SharedCacheMap=captive_SectionObjectPointers_to_SharedCacheMap(SectionObjectPointer);

		if (FileOffset)
			flushed=captive_shared_cache_map_flush(SharedCacheMap,FileOffset->QuadPart,FileOffset->QuadPart+Length);
		else
			flushed=captive_shared_cache_map_flush(SharedCacheMap,0,G_MAXUINT64-1);	/* '-1' for overflow safety */
		}
	else
		flushed=0;

	if (IoStatus) {
		IoStatus->Status=STATUS_SUCCESS;
		IoStatus->Information=flushed;
		}

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcFlushCache: IoStatus->Status=0x%lX,IoStatus->Information=0x%lX",
			(!IoStatus ? -1 : (long)IoStatus->Status),(!IoStatus ? -1 : (long)IoStatus->Information));
}


BOOLEAN CcIsThereDirtyData(IN PVPB Vpb)
{
BOOLEAN r;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcIsThereDirtyData: Vpb=0x%lX",(long)Vpb);

	g_return_val_if_fail(Vpb!=NULL,FALSE);	/* We have just one volume mounted anyway. */

	r=captive_shared_cache_map_is_any_dirty();

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcIsThereDirtyData: r=%d",r);

	return r;
}
