/* $Id: copy.c,v 1.5 2003/09/15 18:02:31 short Exp $
 * reactos Cache Manager (Cc*) CcCopy*() handling of libcaptive
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "privatebcbpin.h"
#include "reactos/ddk/status.h"
#include "captive/macros.h"


/**
 * CcCopyRead:
 * @FileObject: Initialized open #FileObject to map.
 * %NULL value is forbidden.
 * @FileOffset: The @FileObject file offset from where to map the region from.
 * Negative value is forbidden.
 * @Length: Requested length of the region to map from @FileObject.
 * Value %0 is permitted (no effect of this function call).
 * @Wait: Whether disk waiting is permitted for this function.
 * Value %FALSE is currently forbidden by libcaptive as we have no on-demand loading implemented.
 * @Buffer: Address of memory region with already allocated memory of size @Length.
 * This address may not be %PAGE_SIZE aligned.
 * %NULL pointer is forbidden.
 * @IoStatus: #PIO_STATUS_BLOCK to return status of this operation.
 * %NULL pointer is forbidden.
 *
 * Reads the specified region of @FileObject to the given @Buffer.
 * No on-demand loading is in effect.
 *
 * Returns: %TRUE if the region was successfuly filled with @Length bytes.
 * @IoStatus.Status initialized by %STATUS_SUCCESS if successful.
 * @IoStatus.Information initialized by @Length if successful.
 */
BOOLEAN CcCopyRead(IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,IN ULONG Length,IN BOOLEAN Wait,OUT PVOID Buffer,OUT PIO_STATUS_BLOCK IoStatus)
{
CaptiveSharedCacheMapObject *SharedCacheMap;
BOOLEAN r;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcCopyRead: FileObject=0x%lX,FileOffset=0x%lX,Length=0x%lX,Wait=%d,Buffer=0x%lX",
			(long)FileObject,(!FileOffset ? -1 : (long)FileOffset->QuadPart),Length,Wait,(long)Buffer);

	g_return_val_if_fail(FileObject!=NULL,FALSE);
	g_return_val_if_fail(FileOffset!=NULL,FALSE);
	g_return_val_if_fail(Wait==TRUE,FALSE);
	g_return_val_if_fail(Buffer!=NULL,FALSE);
	g_return_val_if_fail(IoStatus!=NULL,FALSE);

	SharedCacheMap=captive_FileObject_to_SharedCacheMap(FileObject);

	captive_shared_cache_map_data_validate_read(SharedCacheMap,FileObject,FileOffset->QuadPart,FileOffset->QuadPart+Length);
	memcpy(Buffer,captive_shared_cache_map_get_buffer(SharedCacheMap)+FileOffset->QuadPart,Length);

	IoStatus->Status=STATUS_SUCCESS;
	IoStatus->Information=Length;
	r=TRUE;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcCopyRead: r=%d,IoStatus->Status=0x%lX,IoStatus->Information=0x%lX",
			r,(!IoStatus ? -1 : (long)IoStatus->Status),(!IoStatus ? -1 : (long)IoStatus->Information));

	return r;
}


/**
 * CcCopyWrite:
 * @FileObject: Initialized open #FileObject to map.
 * %NULL value is forbidden.
 * @FileOffset: The @FileObject file offset from where to map the region from.
 * Negative value is forbidden.
 * @Length: Requested length of the region to map from @FileObject.
 * Value %0 is permitted (no effect of this function call).
 * @Wait: Whether disk waiting is permitted for this function.
 * Value %FALSE is currently forbidden by libcaptive as we have no on-demand loading implemented.
 * @Buffer: Address of memory region with already allocated memory of size @Length.
 * This address may not be %PAGE_SIZE aligned.
 * %NULL pointer is forbidden.
 *
 * Writes the specified region of the given @Buffer to @FileObject.
 *
 * Returns: %TRUE if the region was successfuly written with @Length bytes.
 */
BOOLEAN CcCopyWrite(IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER FileOffset,IN ULONG Length,IN BOOLEAN Wait,IN PVOID Buffer)
{
CaptiveSharedCacheMapObject *SharedCacheMap;
BOOLEAN r;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcCopyWrite: FileObject=0x%lX,FileOffset=0x%lX,Length=0x%lX,Wait=%d,Buffer=0x%lX",
			(long)FileObject,(!FileOffset ? -1 : (long)FileOffset->QuadPart),Length,Wait,(long)Buffer);

	g_return_val_if_fail(FileObject!=NULL,FALSE);
	g_return_val_if_fail(FileOffset!=NULL,FALSE);
	g_return_val_if_fail(Wait==TRUE,FALSE);
	g_return_val_if_fail(Buffer!=NULL,FALSE);

	/* 'FileOffset' may not be PAGE_SIZE aligned */
	/* 'Length' may not be PAGE_SIZE aligned */

	SharedCacheMap=captive_FileObject_to_SharedCacheMap(FileObject);

	captive_shared_cache_map_set_data_valid(SharedCacheMap,FileOffset->QuadPart,FileOffset->QuadPart+Length);
	captive_shared_cache_map_data_validate_read(SharedCacheMap,FileObject,FileOffset->QuadPart,FileOffset->QuadPart+Length);
	memcpy(captive_shared_cache_map_get_buffer(SharedCacheMap)+FileOffset->QuadPart,Buffer,Length);
	captive_shared_cache_map_set_dirty(SharedCacheMap,FileOffset->QuadPart,FileOffset->QuadPart+Length);
	r=TRUE;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcCopyWrite: r=%d",r);

	return r;
}


BOOLEAN CcZeroData(IN PFILE_OBJECT FileObject,IN PLARGE_INTEGER StartOffset,IN PLARGE_INTEGER EndOffset,IN BOOLEAN Wait)
{
CaptiveSharedCacheMapObject *SharedCacheMap,*SharedCacheMap_orig;
BOOLEAN r;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcZeroData: FileObject=0x%lX,StartOffset=0x%lX,EndOffset=0x%lX,Wait=%d",
			(long)FileObject,
			(!StartOffset ? -1 : (long)StartOffset->QuadPart),
			(!EndOffset ? -1 : (long)EndOffset->QuadPart),
			Wait);

	g_return_val_if_fail(FileObject!=NULL,FALSE);
	g_return_val_if_fail(StartOffset!=NULL,FALSE);
	g_return_val_if_fail(EndOffset!=NULL,FALSE);
	g_return_val_if_fail(StartOffset->QuadPart<=EndOffset->QuadPart,FALSE);
	g_return_val_if_fail((EndOffset->QuadPart-StartOffset->QuadPart)
	            ==(size_t)(EndOffset->QuadPart-StartOffset->QuadPart),FALSE);
	/* 'Wait' may be FALSE; used by ext2fsd.sys-v0.10a. */

	/* 'StartOffset' may not be PAGE_SIZE aligned */
	/* 'EndOffset' may not be PAGE_SIZE aligned */

	g_assert(FileObject->SectionObjectPointer);
	if ((SharedCacheMap_orig=FileObject->SectionObjectPointer->SharedCacheMap))
		SharedCacheMap=captive_FileObject_to_SharedCacheMap(FileObject);
	else {
CC_FILE_SIZES FileSizes;
static const CACHE_MANAGER_CALLBACKS CallBacks;

		/* ext2fsd.sys-v0.10a calls us on non-SharedCacheMapped object: */
		FileSizes.AllocationSize=*EndOffset;
		FileSizes.FileSize=*EndOffset;
		FileSizes.ValidDataLength=*EndOffset;
		SharedCacheMap=captive_shared_cache_map_get_ref(
				FileObject,	/* FileObject */
				&FileSizes,	/* FileSizes */
				FALSE,	/* PinAccess */
				&CallBacks,	/* CallBacks */
				NULL);	/* LazyWriterContext */
		}

	captive_shared_cache_map_set_data_valid(SharedCacheMap,StartOffset->QuadPart,EndOffset->QuadPart);
	if (Wait)
		captive_shared_cache_map_data_validate_read(SharedCacheMap,FileObject,StartOffset->QuadPart,EndOffset->QuadPart);
	else {
		captive_shared_cache_map_data_validate_noread(SharedCacheMap,StartOffset->QuadPart,EndOffset->QuadPart);
		}
	memset(captive_shared_cache_map_get_buffer(SharedCacheMap)+StartOffset->QuadPart,0,EndOffset->QuadPart-StartOffset->QuadPart);
	captive_shared_cache_map_set_dirty(SharedCacheMap,StartOffset->QuadPart,EndOffset->QuadPart);
	r=TRUE;

	if (!SharedCacheMap_orig) {
		captive_shared_cache_map_flush(SharedCacheMap,StartOffset->QuadPart,EndOffset->QuadPart);
		g_object_unref(SharedCacheMap);
		/* FIXME: Undo captive_shared_cache_map_get_ref:'Currently we never close it.' */
		g_object_unref(SharedCacheMap);
		}

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcZeroData: r=%d",r);

	return r;
}
