/* $Id: init.c,v 1.13 2003/10/09 10:45:27 short Exp $
 * reactos Cache Manager initialization functions of libcaptive
 * Copyright (C) 2002-2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include <glib/gmessages.h>
#include "reactos/ntos/types.h"
#include "reactos/ddk/iotypes.h"
#include "reactos/ddk/cctypes.h"
#include "sharedcachemap.h"
#include "reactos/ddk/kefuncs.h"


/**
 * CcInitializeCacheMap:
 * @FileObject: Existing file to set callbacks for. Ignored by libcaptive.
 * %NULL value is forbidden.
 * @FileSizes: Some file sizes suggestions. Ignored by libcaptive.
 * %NULL value is forbidden.
 * @PinAccess: CcPin*() functions will be used with @FileObject? Ignored by libcaptive.
 * @CallBacks: Provided callback functions for readahead/writebehind. Ignored by libcaptive.
 * %NULL value is forbidden.
 * @LazyWriterContext: Value passed to functions of @CallBacks to bind with @FileObject.
 * %NULL value is permitted.
 *
 * Provides support of readahead/writebehind in W32. Function should be called
 * by W32 filesystem to offer its functions to W32 kernel. These functions
 * are never called by libcaptive, this call is a NOP in libcaptive.
 *
 * VERIFIED: Double CcInitializeCacheMap() without CcUninitializeCacheMap().
 */
VOID CcInitializeCacheMap(IN PFILE_OBJECT FileObject,
		IN PCC_FILE_SIZES FileSizes,IN BOOLEAN PinAccess,IN PCACHE_MANAGER_CALLBACKS CallBacks,IN PVOID LazyWriterContext)
{
	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcInitializeCacheMap: FileObject=0x%lX,"
					"FileSizes,->AllocationSize=0x%lX,->FileSize=0x%lX,->ValidDataLength=0x%lX,"
					"PinAccess=%d,Callbacks,LazyWriteContext",
			(long)FileObject,
					(!FileSizes ? -1 : (long)FileSizes->AllocationSize.QuadPart),
					(!FileSizes ? -1 : (long)FileSizes->FileSize.QuadPart),
					(!FileSizes ? -1 : (long)FileSizes->ValidDataLength.QuadPart),
					PinAccess);

	g_return_if_fail(FileObject!=NULL);
	g_return_if_fail(FileSizes!=NULL);
	g_return_if_fail(CallBacks!=NULL);

	/* VERIFIED: CcInitializeCacheMap() has already 'SectionObjectPointers' allocated. */
	g_return_if_fail(FileObject->SectionObjectPointer!=NULL);

	captive_shared_cache_map_get_ref(FileObject,FileSizes,PinAccess,CallBacks,LazyWriterContext);

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcInitializeCacheMap");
}


/**
 * CcUninitializeCacheMap:
 * @FileObject: File to close caching for.
 * %NULL value is forbidden.
 * @TruncateSize: Current file size if @FileObject was truncated to it in the meantime.
 * @UninitializeCompleteEvent: Optional event to signal after physical write to disk.
 * %NULL value is permitted.
 *
 * Close the cachine facilities from CcInitializeCacheMap().
 * It is valid to pass @FileObject without being registered by CcInitializeCacheMap().
 *
 * FIXME; What to do with files with dirty blocks? Currently we fail assertion on them
 * although I think W32 would drop such buffers (purge them - not flush them).
 *
 * Returns: %TRUE if the caching was closed successfuly.
 * %FALSE if @FileObject was not registered by CcInitializeCacheMap()
 * or if its #SharedCacheMap is still in use (such as by other #FileObject reference of
 * the same #FCB instance).
 */
BOOLEAN CcUninitializeCacheMap(IN PFILE_OBJECT FileObject,
		IN PLARGE_INTEGER TruncateSize OPTIONAL,IN PCACHE_UNINITIALIZE_EVENT UninitializeCompleteEvent OPTIONAL)
{
CaptiveSharedCacheMapObject *SharedCacheMap;
BOOLEAN r;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcUninitializeCacheMap: FileObject=0x%lX,TruncateSize=0x%lX,UninitializeCompleteEvent=0x%lX",
			(long)FileObject,(!TruncateSize ? -1 : (long)TruncateSize->QuadPart),(long)UninitializeCompleteEvent);

	g_return_val_if_fail(FileObject!=NULL,FALSE);
	/* assert current size ==*TruncateSize if TruncateSize */

	if (FileObject->SectionObjectPointer && !FileObject->SectionObjectPointer->SharedCacheMap)
		r=FALSE;
	else {
		SharedCacheMap=captive_FileObject_to_SharedCacheMap(FileObject);

		/* We may be called without preceding CcInitializeCacheMap(). */
		if (!captive_shared_cache_map_query_w32_ref(SharedCacheMap))
			r=FALSE;
		else {
			captive_shared_cache_map_w32_unref(SharedCacheMap);
			r=(FileObject->SectionObjectPointer->SharedCacheMap==NULL);
			}

		/* FIXME: should we do KePulseEvent? Are we allowed to signal from inside CcUninitializeCacheMap() ? */
		if (UninitializeCompleteEvent)
			KeSetEvent(
					&UninitializeCompleteEvent->Event,	/* Event */
					IO_NO_INCREMENT,	/* Increment */
					FALSE);	/* Wait */
		}

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcUninitializeCacheMap: r=%d",r);

	return r;
}


/**
 * CcSetFileSizes:
 * @FileObject: Initialized open #FileObject to update file sizes of.
 * %NULL value is forbidden.
 * @FileSizes: New file sizes to update cache to.
 * %NULL value is forbidden.
 * 
 * Update cache properties after file sizes were updated.
 * Probably only the exceeding pages need to be unmapped and BCBs updated
 * if FileSizes->AllocationSize gets shrunk.
 *
 * #AllocationSize must not change if any map or pin Bcbs exist.
 */
VOID CcSetFileSizes(IN PFILE_OBJECT FileObject,IN PCC_FILE_SIZES FileSizes)
{
CaptiveSharedCacheMapObject *SharedCacheMap;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcSetFileSizes: FileObject=0x%lX,"
					"FileSizes,->AllocationSize=0x%lX,->FileSize=0x%lX,->ValidDataLength=0x%lX",
			(long)FileObject,
					(!FileSizes ? -1 : (long)FileSizes->AllocationSize.QuadPart),
					(!FileSizes ? -1 : (long)FileSizes->FileSize.QuadPart),
					(!FileSizes ? -1 : (long)FileSizes->ValidDataLength.QuadPart));

	g_return_if_fail(FileObject!=NULL);
	g_return_if_fail(FileSizes!=NULL);

	/* Needed by ext2fsd.sys-v0.10a: */
	if (1
			&& FileObject->SectionObjectPointer
			&& FileObject->SectionObjectPointer->SharedCacheMap) {
		SharedCacheMap=captive_FileObject_to_SharedCacheMap(FileObject);
		captive_shared_cache_map_FileSizes_set(SharedCacheMap,FileSizes);
		}

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcSetFileSizes");
}
