/* $Id: misc.c,v 1.3 2003/09/15 18:02:31 short Exp $
 * reactos Cache Manager (Cc*) misc of libcaptive
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include <glib/gmessages.h>
#include "reactos/ddk/status.h"
#include "reactos/ddk/iotypes.h"
#include "sharedcachemap.h"


/**
 * CcCanIWrite:
 * @FileObject: Initialized open #FileObject to map.
 * %NULL value is forbidden.
 * @BytesToWrite: Amount of data to be asked whether it will be accepted.
 * Value %0 is permitted.
 * @Wait: Whether disk waiting would be permitted during the forthcoming write call.
 * @Retrying: Use %TRUE iff calling this function for the second and further times for one request.
 *
 * Asks cache manager if it would currently accept write request to @FileObject
 * of @BytesToWrite bytes with @Wait condition.
 * libcaptive will always accept any writes. This function is a NOP.
 *
 * Returns: libcaptive always returns %TRUE.
 */
BOOLEAN CcCanIWrite(IN PFILE_OBJECT FileObject,IN ULONG BytesToWrite,IN BOOLEAN Wait,IN BOOLEAN Retrying)
{
BOOLEAN r;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcCanIWrite: FileObject=0x%lX,BytesToWrite=0x%lX,Wait=%d,Retrying=%d",
			(long)FileObject,BytesToWrite,Wait,Retrying);

	g_return_val_if_fail(FileObject!=NULL,FALSE);
	g_return_val_if_fail(Wait==TRUE || Wait==FALSE,FALSE);	/* Prevent 'Wait' upgrade to 'Flags'. */
	g_return_val_if_fail(Retrying==TRUE || Retrying==FALSE,FALSE);

	/* sanity check */
	g_assert(FileObject->SectionObjectPointer!=NULL);
	/* 'FileObject->SectionObjectPointer->SharedCacheMap' may be NULL */

	r=TRUE;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcCanIWrite: r=%d",r);

	return r;
}


/**
 * CcSetReadAheadGranularity:
 * @FileObject: Initialized open #FileObject to map.
 * %NULL value is forbidden.
 * @Granularity: Suggested size of the cache element.
 * Value must be larger or requal to %PAGE_SIZE and it must be even power of two.
 *
 * libcaptive does not implement any caching and therefore this function
 * is a NOP there.
 */
VOID CcSetReadAheadGranularity(IN PFILE_OBJECT FileObject,IN ULONG Granularity)
{
	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcSetReadAheadGranularity: FileObject=0x%lX,Granularity=0x%lX",
			(long)FileObject,Granularity);

	g_return_if_fail(FileObject!=NULL);
	g_return_if_fail(Granularity>=PAGE_SIZE);
	g_return_if_fail((Granularity&(Granularity-1))==0);	/* Power of two */

	/* NOP; no caching by libcaptive */

	/* sanity check */
	captive_FileObject_to_SharedCacheMap(FileObject);

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcSetReadAheadGranularity");
}


/**
 * CcSetAdditionalCacheAttributes:
 * @FileObject: Initialized open #FileObject to map.
 * %NULL value is forbidden.
 * @DisableReadAhead: Read-ahead should not be done by Cache Manager.
 * @DisableWriteBehind: Write-behind should not be done by Cache Manager.
 *
 * libcaptive does not implement any caching and therefore this function
 * is a NOP there.
 */
VOID CcSetAdditionalCacheAttributes(IN PFILE_OBJECT FileObject,IN BOOLEAN DisableReadAhead,IN BOOLEAN DisableWriteBehind)
{
	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcSetAdditionalCacheAttributes: FileObject=0x%lX,DisableReadAhead=%d,DisableWriteBehind=%d",
			(long)FileObject,DisableReadAhead,DisableWriteBehind);

	g_return_if_fail(FileObject!=NULL);

	/* NOP; no caching by libcaptive */

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcSetAdditionalCacheAttributes");
}


VOID FsRtlIncrementCcFastReadNoWait(VOID)
{
	/* FIXME: {{%fs:[0]}+0x4E0}:LONG++ */
}


NTSTATUS CcWaitForCurrentLazyWriterActivity(VOID)
{
NTSTATUS r;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"enter: CcWaitForCurrentLazyWriterActivity");

	r=STATUS_SUCCESS;

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"leave: CcWaitForCurrentLazyWriterActivity: r=0x%lX",(long)r);

	return r;
}
