/* $Id: file-parent.c,v 1.11 2006/01/01 07:24:34 lace Exp $
 * captive vfs 'file' interface to reactos of sandbox parent
 * Copyright (C) 2002-2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "file-parent.h"	/* self */
#include <glib/gmessages.h>
#include "../sandbox/parent-File.h"
#include "vfs-parent.h"
#include "reactos/ntos/types.h"	/* for ULONG */
#include "parent-connector.h"
#include "captive/client.h"	/* for captive_path_normalize() */
#include <ctype.h>


static gpointer captive_file_parent_object_parent_class=NULL;


static GnomeVFSResult captive_file_parent_close(CaptiveFileObject *captive_file_object);
static GnomeVFSResult captive_file_parent_read(CaptiveFileObject *captive_file_object,
		gpointer buffer,GnomeVFSFileSize num_bytes,GnomeVFSFileSize *bytes_read_return);
static GnomeVFSResult captive_file_parent_write(CaptiveFileObject *captive_file_object,
		gconstpointer buffer,GnomeVFSFileSize num_bytes,GnomeVFSFileSize *bytes_written_return);
static GnomeVFSResult captive_file_parent_seek
		(CaptiveFileObject *captive_file_object,GnomeVFSSeekPosition whence,GnomeVFSFileOffset offset);
static GnomeVFSResult captive_file_parent_tell(CaptiveFileObject *captive_file_object,GnomeVFSFileOffset *offset_return);
static GnomeVFSResult captive_file_parent_remove(CaptiveFileObject *captive_file_object);
static GnomeVFSResult captive_file_parent_file_info_get
		(CaptiveFileObject *captive_file_object,CaptiveFileInfoObject **captive_file_info_object_return);
static GnomeVFSResult captive_file_parent_file_info_set
		(CaptiveFileObject *captive_file_object,CaptiveFileInfoObject *captive_file_info_object,GnomeVFSSetFileInfoMask mask);
static GnomeVFSResult captive_file_parent_truncate(CaptiveFileObject *captive_file_object,GnomeVFSFileSize file_size);
static GnomeVFSResult captive_file_parent_move
		(CaptiveFileObject *captive_file_object_old,const gchar *pathname_new,gboolean force_replace);


static void captive_file_parent_object_dispose(CaptiveFileParentObject *captive_file_parent_object)
{
	g_return_if_fail(captive_file_parent_object!=NULL);

	captive_file_parent_close(CAPTIVE_FILE_OBJECT(captive_file_parent_object));	/* errors ignored */

	if (captive_parent_connector_dispose(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object)))
		return;

	G_OBJECT_CLASS(captive_file_parent_object_parent_class)->dispose((GObject *)captive_file_parent_object);
}

static void captive_file_parent_object_finalize(CaptiveFileParentObject *captive_file_parent_object)
{
	g_return_if_fail(captive_file_parent_object!=NULL);

	captive_parent_connector_finalize(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object));

	g_free(captive_file_parent_object->pathname);
	captive_file_parent_object->pathname=NULL;

	G_OBJECT_CLASS(captive_file_parent_object_parent_class)->finalize((GObject *)captive_file_parent_object);
}

static void captive_file_parent_object_class_init(CaptiveFileParentObjectClass *class)
{
GObjectClass *gobject_class=G_OBJECT_CLASS(class);
CaptiveFileObjectClass *captive_file_object_class=CAPTIVE_FILE_OBJECT_CLASS(class);

	captive_file_parent_object_parent_class=g_type_class_ref(g_type_parent(G_TYPE_FROM_CLASS(class)));
	gobject_class->dispose=(void (*)(GObject *object))captive_file_parent_object_dispose;
	gobject_class->finalize=(void (*)(GObject *object))captive_file_parent_object_finalize;

	captive_file_object_class->read=captive_file_parent_read;
	captive_file_object_class->write=captive_file_parent_write;
	captive_file_object_class->seek=captive_file_parent_seek;
	captive_file_object_class->tell=captive_file_parent_tell;
	captive_file_object_class->remove=captive_file_parent_remove;
	captive_file_object_class->file_info_get=captive_file_parent_file_info_get;
	captive_file_object_class->file_info_set=captive_file_parent_file_info_set;
	captive_file_object_class->truncate=captive_file_parent_truncate;
	captive_file_object_class->move=captive_file_parent_move;
}


static GnomeVFSResult (*captive_file_parent_object_captive_parent_connector_open_orig)
		(CaptiveParentConnector *captive_parent_connector);
static GnomeVFSResult captive_file_parent_object_captive_parent_connector_open
		(CaptiveParentConnector *captive_parent_connector);

static GnomeVFSResult captive_file_parent_object_captive_parent_connector_close
		(CaptiveParentConnector *captive_parent_connector);
static GnomeVFSResult (*captive_file_parent_object_captive_parent_connector_close_orig)
		(CaptiveParentConnector *captive_parent_connector);

static G_CONST_RETURN gchar *captive_file_parent_object_captive_parent_connector_get_pathname
		(CaptiveParentConnector *captive_parent_connector);

static void captive_file_parent_object_captive_parent_connector_init(CaptiveParentConnectorIface *captive_parent_connector_iface)
{
	g_return_if_fail(CAPTIVE_IS_PARENT_CONNECTOR_CLASS(captive_parent_connector_iface));

	captive_file_parent_object_captive_parent_connector_open_orig=captive_parent_connector_iface->open;
	captive_parent_connector_iface->open=captive_file_parent_object_captive_parent_connector_open;

	captive_file_parent_object_captive_parent_connector_close_orig=captive_parent_connector_iface->close;
	captive_parent_connector_iface->close=captive_file_parent_object_captive_parent_connector_close;

	captive_parent_connector_iface->get_pathname=captive_file_parent_object_captive_parent_connector_get_pathname;
}


static void captive_file_parent_object_init(CaptiveFileParentObject *captive_file_parent_object)
{
	g_return_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_parent_object));
}


GType captive_file_parent_object_get_type(void)
{
static GType captive_file_parent_object_type=0;

	if (!captive_file_parent_object_type) {
static const GTypeInfo captive_file_parent_object_info={
				sizeof(CaptiveFileParentObjectClass),
				NULL,	/* base_init */
				NULL,	/* base_finalize */
				(GClassInitFunc)captive_file_parent_object_class_init,
				NULL,	/* class_finalize */
				NULL,	/* class_data */
				sizeof(CaptiveFileParentObject),
				5,	/* n_preallocs */
				(GInstanceInitFunc)captive_file_parent_object_init,
				};
static const GInterfaceInfo captive_parent_connector_info={
				(GInterfaceInitFunc)captive_file_parent_object_captive_parent_connector_init,	/* interface_init */
				NULL,	/* interface_finalize */
				NULL,	/* interface_data */
				};

		captive_file_parent_object_type=g_type_register_static(CAPTIVE_FILE_TYPE_OBJECT,
				"CaptiveFileParentObject",&captive_file_parent_object_info,0);
		g_type_add_interface_static(captive_file_parent_object_type,
				CAPTIVE_TYPE_PARENT_CONNECTOR,&captive_parent_connector_info);
		}

	return captive_file_parent_object_type;
}


static void captive_file_parent_init
		(CaptiveFileParentObject *captive_file_parent_object,CaptiveVfsObject *captive_vfs_object)
{
	g_return_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_parent_object));
	g_return_if_fail(CAPTIVE_VFS_PARENT_IS_OBJECT(captive_vfs_object));

	/* Order of captive_file_init() and captive_parent_connector_init()
	 * should not matter as 'vfs' is passed by value to captive_parent_connector_init().
	 */
	captive_file_init(CAPTIVE_FILE_OBJECT(captive_file_parent_object),captive_vfs_object);

	captive_parent_connector_init(
			CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object),	/* captive_parent_connector */
			&captive_file_parent_object->corba_File_object,	/* corba_objectp */
			CAPTIVE_VFS_PARENT_OBJECT(CAPTIVE_FILE_OBJECT(captive_file_parent_object)->vfs));	/* captive_vfs_parent_object */
}

GnomeVFSResult captive_file_parent_new_open(CaptiveFileObject **captive_file_object_return,
		CaptiveVfsObject *captive_vfs_object,const gchar *pathname,GnomeVFSOpenMode mode)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;
gint retried_commit=0;

	g_return_val_if_fail(captive_file_object_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(CAPTIVE_VFS_PARENT_IS_OBJECT(captive_vfs_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(pathname!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=g_object_new(
			CAPTIVE_FILE_PARENT_TYPE_OBJECT,	/* object_type */
			NULL);	/* first_property_name; FIXME: support properties */
	captive_file_parent_object->pathname=g_strdup(pathname);
	captive_file_parent_object->mode=mode;

	captive_file_parent_init(captive_file_parent_object,captive_vfs_object);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_connect(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_new_open(captive_file_parent_object))) {
			if (GNOME_VFS_ERROR_SERVICE_OBSOLETE==r) {
				if (!retried_commit++) {
					if (GNOME_VFS_OK!=(r=captive_vfs_commit(captive_vfs_object)))
						return r;
					retried=0;
					continue;
					}
				}
			if (GNOME_VFS_OK!=r) {
				g_object_unref(captive_file_parent_object);
				*captive_file_object_return=NULL;
				return r;
				}
			*captive_file_object_return=CAPTIVE_FILE_OBJECT(captive_file_parent_object);
			return (*captive_file_parent_object_captive_parent_connector_open_orig)
					(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object));
			}
		} while (!retried++);
	return r;
}


GnomeVFSResult captive_file_parent_new_create(CaptiveFileObject **captive_file_object_return,
		CaptiveVfsObject *captive_vfs_object,const gchar *pathname,GnomeVFSOpenMode mode,gboolean exclusive,guint perm)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;
gint retried_commit=0;

	g_return_val_if_fail(captive_file_object_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(CAPTIVE_VFS_PARENT_IS_OBJECT(captive_vfs_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(pathname!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=g_object_new(
			CAPTIVE_FILE_PARENT_TYPE_OBJECT,	/* object_type */
			NULL);	/* first_property_name; FIXME: support properties */
	captive_file_parent_object->pathname=g_strdup(pathname);
	captive_file_parent_object->mode=mode;

	captive_file_parent_init(captive_file_parent_object,captive_vfs_object);

	if (GNOME_VFS_OK!=(r=captive_parent_connector_connect(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
		return r;

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_connect(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_new_create(captive_file_parent_object,exclusive,perm))) {
			if (GNOME_VFS_ERROR_SERVICE_OBSOLETE==r) {
				if (!retried_commit++) {
					if (GNOME_VFS_OK!=(r=captive_vfs_commit(captive_vfs_object)))
						return r;
					retried=0;
					continue;
					}
				}
			if (GNOME_VFS_OK!=r) {
				g_object_unref(captive_file_parent_object);
				*captive_file_object_return=NULL;
				return r;
				}
			captive_parent_connector_set_dirty(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object));
			*captive_file_object_return=CAPTIVE_FILE_OBJECT(captive_file_parent_object);
			return (*captive_file_parent_object_captive_parent_connector_open_orig)
					(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object));
			}
		} while (!retried++);
	return r;
}


static GnomeVFSResult captive_file_parent_close(CaptiveFileObject *captive_file_object)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;

	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=CAPTIVE_FILE_PARENT_OBJECT(captive_file_object);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_open(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			break;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_parent_connector_close(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			break;
		} while (!retried++);
	return r;
}


static GnomeVFSResult captive_file_parent_object_captive_parent_connector_open
		(CaptiveParentConnector *captive_parent_connector)
{
GnomeVFSResult r;

	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_parent_connector),GNOME_VFS_ERROR_BAD_PARAMETERS);

	if (GNOME_VFS_OK!=(r=captive_sandbox_parent_file_new_open(CAPTIVE_FILE_PARENT_OBJECT(captive_parent_connector))))
		return r;

	return (*captive_file_parent_object_captive_parent_connector_open_orig)(captive_parent_connector);
}


static GnomeVFSResult captive_file_parent_object_captive_parent_connector_close
		(CaptiveParentConnector *captive_parent_connector)
{
GnomeVFSResult r;

	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_parent_connector),GNOME_VFS_ERROR_BAD_PARAMETERS);

	if (GNOME_VFS_OK!=(r=captive_sandbox_parent_file_close(CAPTIVE_FILE_PARENT_OBJECT(captive_parent_connector))))
		return r;

	return (*captive_file_parent_object_captive_parent_connector_close_orig)(captive_parent_connector);
}


static G_CONST_RETURN gchar *captive_file_parent_object_captive_parent_connector_get_pathname
		(CaptiveParentConnector *captive_parent_connector)
{
CaptiveFileParentObject *captive_file_parent_object;

	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),NULL);
	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_parent_connector),NULL);

	captive_file_parent_object=CAPTIVE_FILE_PARENT_OBJECT(captive_parent_connector);

	g_return_val_if_fail(captive_file_parent_object->pathname!=NULL,NULL);

	return captive_file_parent_object->pathname;
}


static GnomeVFSResult captive_file_parent_read(CaptiveFileObject *captive_file_object,
		gpointer buffer,GnomeVFSFileSize num_bytes,GnomeVFSFileSize *bytes_read_return)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;
gint retried_commit=0;

	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(buffer!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(bytes_read_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(num_bytes==(ULONG)num_bytes,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=CAPTIVE_FILE_PARENT_OBJECT(captive_file_object);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_open(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_read(captive_file_parent_object,buffer,num_bytes,bytes_read_return))) {
			if (GNOME_VFS_ERROR_SERVICE_OBSOLETE==r) {
				if (!retried_commit++) {
					if (GNOME_VFS_OK!=(r=captive_vfs_commit(captive_file_object->vfs)))
						return r;
					retried=0;
					continue;
					}
				}
			return r;
			}
		} while (!retried++);
	return r;
}


static GnomeVFSResult captive_file_parent_write(CaptiveFileObject *captive_file_object,
		gconstpointer buffer,GnomeVFSFileSize num_bytes,GnomeVFSFileSize *bytes_written_return)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;
gint retried_commit=0;

	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(buffer!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(bytes_written_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(num_bytes==(ULONG)num_bytes,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=CAPTIVE_FILE_PARENT_OBJECT(captive_file_object);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_open(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_write(captive_file_parent_object,buffer,num_bytes,bytes_written_return))) {
			/* Occured: ExRaiseStatus(STATUS_LOG_FILE_FULL); */
			if (GNOME_VFS_ERROR_SERVICE_OBSOLETE==r) {
				if (!retried_commit++) {
					if (GNOME_VFS_OK!=(r=captive_vfs_commit(captive_file_object->vfs)))
						return r;
					retried=0;
					continue;
					}
				}
			g_assert(*bytes_written_return==num_bytes);	/* Not GNOME_VFS_ERROR_SERVICE_OBSOLETE. */
			if (GNOME_VFS_OK==r)
				captive_parent_connector_set_dirty(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object));
			return r;
			}
		} while (!retried++);
	return r;
}


static GnomeVFSResult captive_file_parent_seek
		(CaptiveFileObject *captive_file_object,GnomeVFSSeekPosition whence,GnomeVFSFileOffset offset)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;

	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=CAPTIVE_FILE_PARENT_OBJECT(captive_file_object);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_open(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_seek(captive_file_parent_object,whence,offset)))
			return r;
		} while (!retried++);
	return r;
}


static GnomeVFSResult captive_file_parent_tell(CaptiveFileObject *captive_file_object,GnomeVFSFileOffset *offset_return)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;

	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(offset_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=CAPTIVE_FILE_PARENT_OBJECT(captive_file_object);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_open(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_tell(captive_file_parent_object,offset_return)))
			return r;
		} while (!retried++);
	return r;
}


static GnomeVFSResult captive_file_parent_remove(CaptiveFileObject *captive_file_object)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;

	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=CAPTIVE_FILE_PARENT_OBJECT(captive_file_object);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_open(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_remove(captive_file_parent_object))) {
			if (GNOME_VFS_OK==r)
				captive_parent_connector_set_dirty(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object));
			return r;
			}
		} while (!retried++);
	return r;
}


static GnomeVFSResult captive_file_parent_file_info_get
		(CaptiveFileObject *captive_file_object,CaptiveFileInfoObject **captive_file_info_object_return)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;

	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(captive_file_info_object_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=CAPTIVE_FILE_PARENT_OBJECT(captive_file_object);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_open(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_file_info_get(captive_file_parent_object,captive_file_info_object_return)))
			return r;
		} while (!retried++);
	return r;
}


static GnomeVFSResult captive_file_parent_file_info_set
		(CaptiveFileObject *captive_file_object,CaptiveFileInfoObject *captive_file_info_object,GnomeVFSSetFileInfoMask mask)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;

	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(CAPTIVE_FILE_INFO_IS_OBJECT(captive_file_info_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=CAPTIVE_FILE_PARENT_OBJECT(captive_file_object);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_open(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_file_info_set(captive_file_parent_object,captive_file_info_object,mask))) {
			if (GNOME_VFS_OK==r)
				captive_parent_connector_set_dirty(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object));
			return r;
			}
		} while (!retried++);
	return r;
}


static GnomeVFSResult captive_file_parent_truncate(CaptiveFileObject *captive_file_object,GnomeVFSFileSize file_size)
{
CaptiveFileParentObject *captive_file_parent_object;
GnomeVFSResult r;
gint retried=0;

	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object=CAPTIVE_FILE_PARENT_OBJECT(captive_file_object);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_open(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_truncate(captive_file_parent_object,file_size))) {
			if (GNOME_VFS_OK==r)
				captive_parent_connector_set_dirty(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object));
			return r;
			}
		} while (!retried++);
	return r;
}


static GnomeVFSResult captive_file_parent_move
		(CaptiveFileObject *captive_file_object_old,const gchar *pathname_new,gboolean force_replace)
{
CaptiveFileParentObject *captive_file_parent_object_old;
GnomeVFSResult r;
gint retried=0;
gchar *chksub_pathname_old_cased,*chksub_pathname_new_cased;	/* case-sensitive version */
gchar *chksub_pathname_old,*chksub_pathname_new,*chksub_s_old,*chksub_s_new;

	g_return_val_if_fail(CAPTIVE_FILE_PARENT_IS_OBJECT(captive_file_object_old),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(pathname_new!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_parent_object_old=CAPTIVE_FILE_PARENT_OBJECT(captive_file_object_old);

	/* Prevent "mv dir dir/subdir" as it is not catched by ntfs.sys of NT-5.1sp1. */
	/* FIXME: Move to 'CaptiveFileSlaveObject' but it has no '->pathname' stored now! */
	/* FIXME: UTF8 may not be compared correctly - we should use g_utf8_collate() */
	chksub_pathname_old_cased=captive_path_normalize(captive_file_parent_object_old->pathname);
	chksub_pathname_new_cased=captive_path_normalize(pathname_new);
	chksub_pathname_old=g_utf8_casefold(chksub_pathname_old_cased,
			-1);	/* len; '\0'-terminated */
	chksub_pathname_new=g_utf8_casefold(chksub_pathname_new_cased,
			-1);	/* len; '\0'-terminated */
	g_free(chksub_pathname_old_cased);
	g_free(chksub_pathname_new_cased);
	for (
			chksub_s_old=chksub_pathname_old,chksub_s_new=chksub_pathname_new;
			*chksub_s_old && *chksub_s_new && tolower(*chksub_s_old)==tolower(*chksub_s_new);
			chksub_s_old++,chksub_s_new++);
	g_assert(chksub_s_old>chksub_pathname_old);
	g_assert(chksub_s_new>chksub_pathname_new);
	if (!*chksub_s_old && (!*chksub_s_new || *chksub_s_new=='/'))
		return GNOME_VFS_ERROR_DIRECTORY_BUSY;
	if (!*chksub_s_new && (!*chksub_s_old || *chksub_s_old=='/'))
		return GNOME_VFS_ERROR_DIRECTORY_BUSY;
	g_free(chksub_pathname_old);
	g_free(chksub_pathname_new);

	do {
		if (GNOME_VFS_OK!=(r=captive_parent_connector_open(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object_old))))
			return r;
		if (GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE
				!=(r=captive_sandbox_parent_file_move(captive_file_parent_object_old,pathname_new,force_replace))) {
			if (GNOME_VFS_OK==r) {
				g_free(captive_file_parent_object_old->pathname);
				captive_file_parent_object_old->pathname=g_strdup(pathname_new);
				captive_parent_connector_set_dirty(CAPTIVE_PARENT_CONNECTOR(captive_file_parent_object_old));
				}
			return r;
			}
		} while (!retried++);
	return r;
}
