/* $Id: file-slave.c,v 1.12 2006/01/01 07:24:34 lace Exp $
 * captive vfs 'file' interface to reactos
 * Copyright (C) 2002-2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "file-slave.h"	/* self */
#include "captive/client-file.h"	/* for CaptiveFileObject */
#include "file.h"	/* for CaptiveFileObject priv */
#include "directory.h"
#include "lib.h"
#include <glib/gmessages.h>
#include "captive/unicode.h"
#include "reactos/ntos/types.h"	/* for HANDLE */
#include "reactos/ddk/iotypes.h"	/* for IO_STATUS_BLOCK */
#include "reactos/ddk/iofuncs.h"	/* for IoCreateFile() */
#include "result.h"
#include "captive/client-directory.h"
#include "reactos/ddk/obfuncs.h"	/* for ObReferenceObjectByHandle() */
#include "captive/macros.h"
#include "captive/leave.h"
#include "captive/usecount.h"
#include "vfs.h"
#include "vfs-slave.h"
#include "../cc/sharedcachemap.h"
#include "captive/rtl-time.h"


/* Config: */
#define MAX_FILE_READ   0x8000000	/* FIXME: Workaround memory consumption for non-journalled fastfat.sys */
#define MAX_FILE_WRITTEN 0x100000	/* FIXME: Workaround memory consumption for non-journalled fastfat.sys */


struct _CaptiveFileSlaveObject {
	CaptiveFileObject parent_instance;

	/*< private >*/
	HANDLE file_Handle;
	GnomeVFSFileOffset offset;
	};
struct _CaptiveFileSlaveObjectClass {
	CaptiveFileObjectClass parent_class;
	};


static gpointer captive_file_slave_object_parent_class=NULL;


static GnomeVFSResult captive_file_slave_close(CaptiveFileObject *captive_file_object);
static GnomeVFSResult captive_file_slave_read(CaptiveFileObject *captive_file_object,
		gpointer buffer,GnomeVFSFileSize num_bytes,GnomeVFSFileSize *bytes_read_return);
static GnomeVFSResult captive_file_slave_write(CaptiveFileObject *captive_file_object,
		gconstpointer buffer,GnomeVFSFileSize num_bytes,GnomeVFSFileSize *bytes_written_return);
static GnomeVFSResult captive_file_slave_seek
		(CaptiveFileObject *captive_file_object,GnomeVFSSeekPosition whence,GnomeVFSFileOffset offset);
static GnomeVFSResult captive_file_slave_tell(CaptiveFileObject *captive_file_object,GnomeVFSFileOffset *offset_return);
static GnomeVFSResult captive_file_slave_remove(CaptiveFileObject *captive_file_object);
static GnomeVFSResult captive_file_slave_file_info_get
		(CaptiveFileObject *captive_file_object,CaptiveFileInfoObject **captive_file_info_object_return);
static GnomeVFSResult captive_file_slave_file_info_set
		(CaptiveFileObject *captive_file_object,CaptiveFileInfoObject *captive_file_info_object,GnomeVFSSetFileInfoMask mask);
static GnomeVFSResult captive_file_slave_truncate(CaptiveFileObject *captive_file_object,GnomeVFSFileSize file_size);
static GnomeVFSResult captive_file_slave_move
		(CaptiveFileObject *captive_file_object_old,const gchar *pathname_new,gboolean force_replace);


static void captive_file_slave_object_finalize(CaptiveFileSlaveObject *captive_file_slave_object)
{
	g_return_if_fail(captive_file_slave_object!=NULL);

	captive_file_slave_close(CAPTIVE_FILE_OBJECT(captive_file_slave_object));	/* errors ignored */

	G_OBJECT_CLASS(captive_file_slave_object_parent_class)->finalize((GObject *)captive_file_slave_object);
}


static void captive_file_slave_object_class_init(CaptiveFileSlaveObjectClass *class)
{
GObjectClass *gobject_class=G_OBJECT_CLASS(class);
CaptiveFileObjectClass *captive_file_object_class=CAPTIVE_FILE_OBJECT_CLASS(class);

	captive_file_slave_object_parent_class=g_type_class_ref(g_type_parent(G_TYPE_FROM_CLASS(class)));
	gobject_class->finalize=(void (*)(GObject *object))captive_file_slave_object_finalize;

	captive_file_object_class->read=captive_file_slave_read;
	captive_file_object_class->write=captive_file_slave_write;
	captive_file_object_class->seek=captive_file_slave_seek;
	captive_file_object_class->tell=captive_file_slave_tell;
	captive_file_object_class->remove=captive_file_slave_remove;
	captive_file_object_class->file_info_get=captive_file_slave_file_info_get;
	captive_file_object_class->file_info_set=captive_file_slave_file_info_set;
	captive_file_object_class->truncate=captive_file_slave_truncate;
	captive_file_object_class->move=captive_file_slave_move;
}


static void captive_file_slave_object_init(CaptiveFileSlaveObject *captive_file_slave_object)
{
	captive_file_slave_object->file_Handle=NULL;
}


GType captive_file_slave_object_get_type(void)
{
static GType captive_file_slave_object_type=0;

	if (!captive_file_slave_object_type) {
static const GTypeInfo captive_file_slave_object_info={
				sizeof(CaptiveFileSlaveObjectClass),
				NULL,	/* base_init */
				NULL,	/* base_finalize */
				(GClassInitFunc)captive_file_slave_object_class_init,
				NULL,	/* class_finalize */
				NULL,	/* class_data */
				sizeof(CaptiveFileSlaveObject),
				5,	/* n_preallocs */
				(GInstanceInitFunc)captive_file_slave_object_init,
				};

		captive_file_slave_object_type=g_type_register_static(CAPTIVE_FILE_TYPE_OBJECT,
				"CaptiveFileSlaveObject",&captive_file_slave_object_info,0);
		}

	return captive_file_slave_object_type;
}


static GnomeVFSResult captive_file_slave_new_internal(CaptiveFileSlaveObject *captive_file_slave_object,
		const gchar *pathname,GnomeVFSOpenMode mode,gboolean create,gboolean create_exclusive,guint create_perm)
{
IO_STATUS_BLOCK file_IoStatusBlock;
GnomeVFSResult errvfsresult;
OBJECT_ATTRIBUTES file_ObjectAttributes;
HANDLE file_Handle;
NTSTATUS err;

	g_return_val_if_fail(captive_file_slave_object!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(pathname!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	if (captive_shared_cache_map_restart_needed())
		return GNOME_VFS_ERROR_SERVICE_OBSOLETE;

	errvfsresult=captive_ObjectAttributes_init(pathname,&file_ObjectAttributes);
	g_return_val_if_fail(errvfsresult==GNOME_VFS_OK,errvfsresult);
	
	/* open */
	err=IoCreateFile(
			&file_Handle,	/* FileHandle */
			0
					/* sniffed: | SYNCHRONIZE	*/
					|(!(mode&GNOME_VFS_OPEN_READ ) ? 0 : FILE_READ_DATA)
					|(!(mode&GNOME_VFS_OPEN_WRITE) ? 0 : FILE_WRITE_DATA | FILE_APPEND_DATA | FILE_WRITE_ATTRIBUTES)
					|(  mode!=0                    ? 0 : FILE_READ_ATTRIBUTES)
					,	/* DesiredAccess */
			&file_ObjectAttributes,	/* ObjectAttributes */
			&file_IoStatusBlock,	/* IoStatusBlock */
			NULL,	/* AllocationSize; ignored for open */
			(!create || create_perm&0200 ? FILE_ATTRIBUTE_NORMAL: FILE_ATTRIBUTE_READONLY),	/* FileAttributes; ignored for open */
			(!create || !create_exclusive	/* ShareAccess; 0 means exclusive */
					? (FILE_SHARE_READ | FILE_SHARE_WRITE | FILE_SHARE_DELETE)
					: 0),
			(!create ? FILE_OPEN : FILE_CREATE),	/* CreateDisposition */
			/* FILE_SYNCHRONOUS_IO_{,NON}ALERT: We need to allow W32 filesystem
				 * any waits to not to let it return STATUS_CANT_WAIT us.
				 * Alertability should have only effect on asynchronous events
				 * from KeWaitForSingleObject() by setting/clearing its parameter 'Alertable'.
				 */
			FILE_SYNCHRONOUS_IO_ALERT	/* CreateOptions */
					/* sniffed: | FILE_DIRECTORY_FILE */
					/* sniffed: | FILE_OPEN_FOR_BACKUP_INTENT */
					,
			NULL,	/* EaBuffer */
			0,	/* EaLength */
			CreateFileTypeNone,	/* CreateFileType */
			NULL,	/* ExtraCreateParameters */
			0);	/* Options */
	g_free(file_ObjectAttributes.ObjectName);	/* left from captive_file_slave_uri_parent_init() */
	g_return_val_if_fail(NT_SUCCESS(err)==NT_SUCCESS(file_IoStatusBlock.Status),GNOME_VFS_ERROR_GENERIC);
	if (GNOME_VFS_OK!=(errvfsresult=captive_NTSTATUS_to_GnomeVFSResult(err)))
		return errvfsresult;
	g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);
	g_return_val_if_fail(file_IoStatusBlock.Information
					==(!create ? FILE_OPENED : FILE_CREATED),
			GNOME_VFS_ERROR_GENERIC);

	captive_file_slave_object->file_Handle=file_Handle;

	return GNOME_VFS_OK;
}


GnomeVFSResult captive_file_slave_new_open(CaptiveFileObject **captive_file_object_return,
		CaptiveVfsObject *captive_vfs_object,const gchar *pathname,GnomeVFSOpenMode mode)
{
GnomeVFSResult r;
CaptiveFileSlaveObject *captive_file_slave_object;

	g_return_val_if_fail(captive_file_object_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(CAPTIVE_VFS_SLAVE_IS_OBJECT(captive_vfs_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(pathname!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=g_object_new(
			CAPTIVE_FILE_SLAVE_TYPE_OBJECT,	/* object_type */
			NULL);	/* first_property_name; FIXME: support properties */

	captive_file_init(CAPTIVE_FILE_OBJECT(captive_file_slave_object),captive_vfs_object);

	*captive_file_object_return=CAPTIVE_FILE_OBJECT(captive_file_slave_object);

	r=captive_file_slave_new_internal(captive_file_slave_object,pathname,mode,
			FALSE,	/* create */
			FALSE,	/* create_exclusive; ignored */
			0);	/* create_perm; ignored */

	captive_leave();
	if (r==GNOME_VFS_OK)
		captive_usecount(+1);
	else {
		g_object_unref(captive_file_slave_object);
		*captive_file_object_return=NULL;
		}
	return r;
}


GnomeVFSResult captive_file_slave_new_create(CaptiveFileObject **captive_file_object_return,
		CaptiveVfsObject *captive_vfs_object,const gchar *pathname,GnomeVFSOpenMode mode,gboolean exclusive,guint perm)
{
GnomeVFSResult r;
CaptiveFileSlaveObject *captive_file_slave_object;

	g_return_val_if_fail(captive_file_object_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(CAPTIVE_VFS_SLAVE_IS_OBJECT(captive_vfs_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(pathname!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=g_object_new(
			CAPTIVE_FILE_SLAVE_TYPE_OBJECT,	/* object_type */
			NULL);	/* first_property_name; FIXME: support properties */

	captive_file_init(CAPTIVE_FILE_OBJECT(captive_file_slave_object),captive_vfs_object);

	*captive_file_object_return=CAPTIVE_FILE_OBJECT(captive_file_slave_object);

	r=captive_file_slave_new_internal(captive_file_slave_object,pathname,mode,
			TRUE,	/* create */
			exclusive,	/* create_exclusive */
			perm);	/* create_perm */

	captive_leave();
	if (r==GNOME_VFS_OK)
		captive_usecount(+1);
	else {
		g_object_unref(captive_file_slave_object);
		*captive_file_object_return=NULL;
		}
	return r;
}


static GnomeVFSResult captive_file_slave_close(CaptiveFileObject *captive_file_object)
{
CaptiveFileSlaveObject *captive_file_slave_object;
NTSTATUS err;
HANDLE file_Handle;

	g_return_val_if_fail(CAPTIVE_FILE_SLAVE_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=CAPTIVE_FILE_SLAVE_OBJECT(captive_file_object);

	if (captive_file_slave_object->file_Handle!=NULL) {	/* not yet already closed */
		captive_usecount(-1);	/* close() errors notwithstanding */

		file_Handle=captive_file_slave_object->file_Handle;
		captive_file_slave_object->file_Handle=NULL;
		err=NtClose(file_Handle);
		g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);
		}

	captive_leave();
	return GNOME_VFS_OK;
}


GnomeVFSResult captive_file_slave_read(CaptiveFileObject *captive_file_object,
		gpointer buffer,GnomeVFSFileSize num_bytes,GnomeVFSFileSize *bytes_read_return)
{
CaptiveFileSlaveObject *captive_file_slave_object;
NTSTATUS err;
IO_STATUS_BLOCK file_IoStatusBlock;
LARGE_INTEGER file_offset;
GnomeVFSResult errvfsresult;
static GnomeVFSFileSize total_read=0;

	g_return_val_if_fail(CAPTIVE_FILE_SLAVE_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(buffer!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(bytes_read_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(num_bytes==(ULONG)num_bytes,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=CAPTIVE_FILE_SLAVE_OBJECT(captive_file_object);

	g_return_val_if_fail(captive_file_slave_object->file_Handle!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	/* FIXME: Workaround memory consumption for non-journalled fastfat.sys */
	if (total_read>=MAX_FILE_READ && captive_options->sandbox)
		return GNOME_VFS_ERROR_SERVICE_OBSOLETE;
	total_read+=num_bytes;

	file_offset.QuadPart=captive_file_slave_object->offset;
	err=NtReadFile(
			captive_file_slave_object->file_Handle,	/* FileHandle */
			NULL,	/* EventHandle; completion signalling; optional */
			NULL,	/* ApcRoutine; optional */
			NULL,	/* ApcContext; optional */
			&file_IoStatusBlock,	/* IoStatusBlock */
			buffer,	/* Buffer */
			num_bytes,	/* Length */
			&file_offset,	/* ByteOffset */
			NULL);	/* Key; NULL means no file locking key */
	if (err==STATUS_END_OF_FILE) {
		g_return_val_if_fail(file_IoStatusBlock.Status==STATUS_END_OF_FILE,GNOME_VFS_ERROR_GENERIC);
		g_return_val_if_fail(file_IoStatusBlock.Information==0,GNOME_VFS_ERROR_GENERIC);
		*bytes_read_return=0;
		return GNOME_VFS_ERROR_EOF;
		}
	g_return_val_if_fail(file_IoStatusBlock.Information>0,GNOME_VFS_ERROR_GENERIC);	/* if not STATUS_END_OF_FILE... */
	if (GNOME_VFS_OK!=(errvfsresult=captive_NTSTATUS_to_GnomeVFSResult(err)))
		return errvfsresult;
	g_return_val_if_fail(NT_SUCCESS(err)==NT_SUCCESS(file_IoStatusBlock.Status),GNOME_VFS_ERROR_GENERIC);
	g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);
	g_return_val_if_fail(file_IoStatusBlock.Information<=num_bytes,GNOME_VFS_ERROR_GENERIC);

	captive_file_slave_object->offset+=file_IoStatusBlock.Information;
	*bytes_read_return=file_IoStatusBlock.Information;
	captive_leave();
	return GNOME_VFS_OK;
}


GnomeVFSResult captive_file_slave_write(CaptiveFileObject *captive_file_object,
		gconstpointer buffer,GnomeVFSFileSize num_bytes,GnomeVFSFileSize *bytes_written_return)
{
CaptiveFileSlaveObject *captive_file_slave_object;
NTSTATUS err;
IO_STATUS_BLOCK file_IoStatusBlock;
LARGE_INTEGER file_offset;
CaptiveFileInfoObject *captive_file_info_object;
GnomeVFSFileSize endoffile_wanted;
GnomeVFSResult errvfsresult;
static GnomeVFSFileSize total_written=0;
gboolean extension_needed;

	g_return_val_if_fail(CAPTIVE_FILE_SLAVE_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(buffer!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(bytes_written_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(num_bytes==(ULONG)num_bytes,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=CAPTIVE_FILE_SLAVE_OBJECT(captive_file_object);

	g_return_val_if_fail(captive_file_slave_object->file_Handle!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	/* FIXME: Workaround memory consumption for non-journalled fastfat.sys */
	if (total_written>=MAX_FILE_WRITTEN && captive_options->sandbox)
		return GNOME_VFS_ERROR_SERVICE_OBSOLETE;
	total_written+=num_bytes;

	if ((GnomeVFSFileOffset)(captive_file_slave_object->offset+num_bytes) < captive_file_slave_object->offset)
		return GNOME_VFS_ERROR_TOO_BIG;
	endoffile_wanted=captive_file_slave_object->offset+num_bytes;
	if (GNOME_VFS_OK!=(errvfsresult=captive_file_slave_file_info_get(
			CAPTIVE_FILE_OBJECT(captive_file_slave_object),&captive_file_info_object)))
		return errvfsresult;
	g_assert(captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_SIZE);
	extension_needed=(captive_file_info_object->p.size<endoffile_wanted);
	g_object_unref(captive_file_info_object);
	if (extension_needed) {
		if (GNOME_VFS_OK!=(errvfsresult=captive_file_slave_truncate(
				CAPTIVE_FILE_OBJECT(captive_file_slave_object),endoffile_wanted)))
			return errvfsresult;
		}

	file_offset.QuadPart=captive_file_slave_object->offset;
	err=NtWriteFile(
			captive_file_slave_object->file_Handle,	/* FileHandle */
			NULL,	/* Event; completion signalling; optional */
			NULL,	/* ApcRoutine; optional */
			NULL,	/* ApcContext; optional */
			&file_IoStatusBlock,	/* IoStatusBlock */
			(gpointer /* de-const */ )buffer,	/* Buffer */
			num_bytes,	/* Length */
			&file_offset,	/* ByteOffset */
			NULL);	/* Key; NULL means no file locking key */
	g_return_val_if_fail(NT_SUCCESS(err)==NT_SUCCESS(file_IoStatusBlock.Status),GNOME_VFS_ERROR_GENERIC);
	g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);
	if (!file_IoStatusBlock.Information) {
		/* FIXME: Check ExRaiseStatus(STATUS_LOG_FILE_FULL);
		 * really occured. We expect so and we will remount the volume.
		 */
		*bytes_written_return=0;
		return GNOME_VFS_ERROR_SERVICE_OBSOLETE;
		}
	g_return_val_if_fail(file_IoStatusBlock.Information==num_bytes,GNOME_VFS_ERROR_GENERIC);

	captive_file_slave_object->offset+=file_IoStatusBlock.Information;
	*bytes_written_return=file_IoStatusBlock.Information;
	captive_leave();
	return GNOME_VFS_OK;
}


GnomeVFSResult captive_file_slave_seek
		(CaptiveFileObject *captive_file_object,GnomeVFSSeekPosition whence,GnomeVFSFileOffset offset)
{
CaptiveFileSlaveObject *captive_file_slave_object;

	g_return_val_if_fail(CAPTIVE_FILE_SLAVE_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=CAPTIVE_FILE_SLAVE_OBJECT(captive_file_object);

	g_return_val_if_fail(captive_file_slave_object->file_Handle!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	switch (whence) {
		case GNOME_VFS_SEEK_START:
			captive_file_slave_object->offset=offset;
			break;
		case GNOME_VFS_SEEK_CURRENT:
			if (0
					|| (offset>0 && (captive_file_slave_object->offset+offset)<captive_file_slave_object->offset)
					|| (offset<0 && (captive_file_slave_object->offset+offset)>captive_file_slave_object->offset))
				return GNOME_VFS_ERROR_BAD_PARAMETERS;
			captive_file_slave_object->offset+=offset;
			break;
		case GNOME_VFS_SEEK_END:
			g_assert_not_reached();	/* NOT IMPLEMENTED YET */
			break;
		default:
			g_return_val_if_reached(GNOME_VFS_ERROR_BAD_PARAMETERS);
		}

	captive_leave();
	return GNOME_VFS_OK;
}


GnomeVFSResult captive_file_slave_tell(CaptiveFileObject *captive_file_object,GnomeVFSFileOffset *offset_return)
{
CaptiveFileSlaveObject *captive_file_slave_object;

	g_return_val_if_fail(CAPTIVE_FILE_SLAVE_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(offset_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=CAPTIVE_FILE_SLAVE_OBJECT(captive_file_object);

	g_return_val_if_fail(captive_file_slave_object->file_Handle!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	*offset_return=captive_file_slave_object->offset;
	captive_leave();
	return GNOME_VFS_OK;
}


GnomeVFSResult captive_file_slave_remove(CaptiveFileObject *captive_file_object)
{
CaptiveFileSlaveObject *captive_file_slave_object;
NTSTATUS err;
FILE_DISPOSITION_INFORMATION FileDispositionInformation_struct;
IO_STATUS_BLOCK file_IoStatusBlock;
GnomeVFSResult errvfsresult;

	g_return_val_if_fail(CAPTIVE_FILE_SLAVE_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=CAPTIVE_FILE_SLAVE_OBJECT(captive_file_object);

	g_return_val_if_fail(captive_file_slave_object->file_Handle!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	FileDispositionInformation_struct.DoDeleteFile=TRUE;

	err=NtSetInformationFile(
			captive_file_slave_object->file_Handle,	/* FileHandle */
			&file_IoStatusBlock,	/* IoStatusBlock */
			&FileDispositionInformation_struct,	/* FileInformation */
			sizeof(FileDispositionInformation_struct),	/* Length */
			FileDispositionInformation);	/* FileInformationClass */
	if (GNOME_VFS_OK!=(errvfsresult=captive_NTSTATUS_to_GnomeVFSResult(err)))
		return errvfsresult;
	g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);

	captive_leave();
	return GNOME_VFS_OK;
}


/* Use 'FileNameInformationStruct' identifier instead of the logical 'FileNameInformation'
 * to prevent override of enum member 'FileNameInformation'
 */
static GnomeVFSResult FileNameInformationStruct_to_CaptiveFileInfoObject_in_place
		(CaptiveFileInfoObject *captive_file_info_object,FILE_NAME_INFORMATION *FileNameInformationStruct)
{
UNICODE_STRING FileName_UnicodeString;

	g_return_val_if_fail(CAPTIVE_FILE_INFO_IS_OBJECT(captive_file_info_object),GNOME_VFS_ERROR_GENERIC);
	g_return_val_if_fail(FileNameInformationStruct!=NULL,GNOME_VFS_ERROR_GENERIC);

	FileName_UnicodeString.Length=FileNameInformationStruct->FileNameLength;
	FileName_UnicodeString.MaximumLength=FileNameInformationStruct->FileNameLength
			+sizeof(*FileNameInformationStruct->FileName);	/* 0-terminator */

#if 0	/* 'IoStatusBlock->Information' is sometimes weird, do not check it */
	g_assert((gpointer)(((char *)FileNameInformationStruct->FileName)+FileName_UnicodeString.Length)
			<=(gpointer)(((char *)FileAllInformationStruct)+IoStatusBlock->Information));
					/* ensure we fit below '->IoStatusBlock->Information' at least without the 0-terminator */
#endif

	FileNameInformationStruct->FileName[FileNameInformationStruct->FileNameLength
			/sizeof(*FileNameInformationStruct->FileName)]=0;	/* 0-terminate it */
	FileName_UnicodeString.Buffer=FileNameInformationStruct->FileName;
	captive_file_info_object->p.name=captive_UnicodeString_to_utf8_malloc(&FileName_UnicodeString);
	/* '->name' assumed for 'captive_file_info_object->p.valid_fields' */

	return GNOME_VFS_OK;
}


/* Use 'FileBasicInformationStruct' identifier instead of the logical 'FileBasicInformation'
 * to prevent override of enum member 'FileBasicInformation'
 */
static GnomeVFSResult FileBasicInformationStruct_to_CaptiveFileInfoObject_in_place
		(CaptiveFileInfoObject *captive_file_info_object,FILE_BASIC_INFORMATION *FileBasicInformationStruct)
{
GnomeVFSResult errvfsresult;

	g_return_val_if_fail(CAPTIVE_FILE_INFO_IS_OBJECT(captive_file_info_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	/* FIXME: What is 'FILE_ATTRIBUTE_NORMAL'? */
	switch (FileBasicInformationStruct->FileAttributes & (0
			| FILE_ATTRIBUTE_DIRECTORY
			| FILE_ATTRIBUTE_DEVICE)) {
		case 0:                        captive_file_info_object->p.type=GNOME_VFS_FILE_TYPE_REGULAR;   break;
		case FILE_ATTRIBUTE_DIRECTORY: captive_file_info_object->p.type=GNOME_VFS_FILE_TYPE_DIRECTORY; break;
		case FILE_ATTRIBUTE_DEVICE:    captive_file_info_object->p.type=GNOME_VFS_FILE_TYPE_SOCKET;
			/* or GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE or GNOME_VFS_FILE_TYPE_BLOCK_DEVICE ? */
			break;
		default:                       captive_file_info_object->p.type=GNOME_VFS_FILE_TYPE_UNKNOWN;   break;
		}
	captive_file_info_object->p.valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_TYPE;

	/* we use 0600 for r/w files, 0400 for FILE_ATTRIBUTE_READONLY */
	captive_file_info_object->p.permissions=GNOME_VFS_PERM_USER_READ;
	if (captive_file_info_object->p.type==GNOME_VFS_FILE_TYPE_DIRECTORY)
		captive_file_info_object->p.permissions|=GNOME_VFS_PERM_USER_EXEC;
	if (!(FileBasicInformationStruct->FileAttributes & FILE_ATTRIBUTE_READONLY))
		captive_file_info_object->p.permissions|=GNOME_VFS_PERM_USER_WRITE;
	captive_file_info_object->p.valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS;

	if (FileBasicInformationStruct->LastAccessTime.QuadPart) {	/* it may be 0 if not set */
struct timespec timespec;

		errvfsresult=Time_to_timespec(&timespec,&FileBasicInformationStruct->LastAccessTime);
		g_return_val_if_fail(GNOME_VFS_OK==errvfsresult,errvfsresult);
		captive_file_info_object->p.atime=timespec.tv_sec;
		captive_file_info_object->atime_nsec=timespec.tv_nsec;
		captive_file_info_object->p.valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_ATIME;
		}

	/* it may be 0 if not set */
	if (FileBasicInformationStruct->LastWriteTime.QuadPart || FileBasicInformationStruct->ChangeTime.QuadPart) {
struct timespec timespec;

		errvfsresult=Time_to_timespec(&timespec,
				/* take the more recent (==bigger) time: */
				(FileBasicInformationStruct->LastWriteTime.QuadPart > FileBasicInformationStruct->ChangeTime.QuadPart
						? &FileBasicInformationStruct->LastWriteTime : &FileBasicInformationStruct->ChangeTime));
		g_return_val_if_fail(GNOME_VFS_OK==errvfsresult,errvfsresult);
		captive_file_info_object->p.mtime=timespec.tv_sec;
		captive_file_info_object->mtime_nsec=timespec.tv_nsec;
		captive_file_info_object->p.valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_MTIME;
		}

	if (FileBasicInformationStruct->CreationTime.QuadPart) {	/* it may be 0 if not set */
struct timespec timespec;

		errvfsresult=Time_to_timespec(&timespec,&FileBasicInformationStruct->CreationTime);
		g_return_val_if_fail(GNOME_VFS_OK==errvfsresult,errvfsresult);
		captive_file_info_object->p.ctime=timespec.tv_sec;
		captive_file_info_object->ctime_nsec=timespec.tv_nsec;
		captive_file_info_object->p.valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_CTIME;
		}

	return GNOME_VFS_OK;
}


/* Use 'FileStandardInformationStruct' identifier instead of the logical 'FileStandardInformation'
 * to prevent override of enum member 'FileStandardInformation'
 */
static GnomeVFSResult FileStandardInformationStruct_to_CaptiveFileInfoObject_in_place
		(CaptiveFileInfoObject *captive_file_info_object,FILE_STANDARD_INFORMATION *FileStandardInformationStruct)
{
	g_return_val_if_fail(CAPTIVE_FILE_INFO_IS_OBJECT(captive_file_info_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_info_object->p.size=FileStandardInformationStruct->EndOfFile.QuadPart;
	captive_file_info_object->p.valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_SIZE;

	captive_file_info_object->p.block_count=FileStandardInformationStruct->AllocationSize.QuadPart/512;
	captive_file_info_object->p.valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_BLOCK_COUNT;

	return GNOME_VFS_OK;
}


GnomeVFSResult captive_file_slave_file_info_get(CaptiveFileObject *captive_file_object,
		CaptiveFileInfoObject **captive_file_info_object_return)
{
CaptiveFileInfoObject *captive_file_info_object;
CaptiveFileSlaveObject *captive_file_slave_object;
NTSTATUS err;
IO_STATUS_BLOCK file_IoStatusBlock;
FILE_NAME_INFORMATION *FileNameInformationStruct;
FILE_BASIC_INFORMATION *FileBasicInformationStruct;
FILE_STANDARD_INFORMATION *FileStandardInformationStruct;
GnomeVFSResult errvfsresult;
char QueryFile_buf[sizeof(FILE_NAME_INFORMATION)
		+0x1000 /* max 'FileName' length, 255 should be enough */ * sizeof(WCHAR /* *FILE_ALL_INFORMATION.NameInformation.FileName */ )];

	g_return_val_if_fail(CAPTIVE_FILE_SLAVE_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(captive_file_info_object_return!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=CAPTIVE_FILE_SLAVE_OBJECT(captive_file_object);

	g_return_val_if_fail(captive_file_slave_object->file_Handle!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	errvfsresult=captive_file_info_object_new(&captive_file_info_object);
	g_return_val_if_fail(GNOME_VFS_OK==errvfsresult,errvfsresult);

	/* ntfs.sys of NT-5.1sp1 has *(FILE_BASIC_INFORMATION *)FILE_ALL_INFORMATION.BasicInformation
	 * size 0x28 although reactos includes and all the information says 0x24.
	 * fastfat.sys of NT-5.1sp1 also has the 'official' sizeof 0x24.
	 * It would be a bit of heuristic to correctly detect its expected sizeof.
	 */

	captive_file_info_object->p.flags=GNOME_VFS_FILE_FLAGS_LOCAL;
	captive_file_info_object->p.valid_fields|=GNOME_VFS_FILE_INFO_FIELDS_FLAGS;

	/* query NameInformation */
	FileNameInformationStruct=(void *)QueryFile_buf;
	err=NtQueryInformationFile(
			captive_file_slave_object->file_Handle,	/* FileHandle */
			&file_IoStatusBlock,	/* IoStatusBlock */
			(gpointer)QueryFile_buf,	/* FileInformation */
			sizeof(QueryFile_buf)	/* Length */
					-sizeof(*FileNameInformationStruct->FileName),	/* reserve space for a possible 0-terminator */
			FileNameInformation);	/* FileInformationClass; =>FILE_NAME_INFORMATION */
	if (GNOME_VFS_OK!=(errvfsresult=captive_NTSTATUS_to_GnomeVFSResult(err)))
		return errvfsresult;
	g_return_val_if_fail(NT_SUCCESS(err)==NT_SUCCESS(file_IoStatusBlock.Status),GNOME_VFS_ERROR_GENERIC);
	g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);
	errvfsresult=FileNameInformationStruct_to_CaptiveFileInfoObject_in_place(captive_file_info_object,
			FileNameInformationStruct);
	g_return_val_if_fail(errvfsresult==GNOME_VFS_OK,errvfsresult);

	/* query BasicInformation */
	FileBasicInformationStruct=(void *)QueryFile_buf;
	err=NtQueryInformationFile(
			captive_file_slave_object->file_Handle,	/* FileHandle */
			&file_IoStatusBlock,	/* IoStatusBlock */
			(gpointer)QueryFile_buf,	/* FileInformation */
			sizeof(QueryFile_buf),	/* Length */
			FileBasicInformation);	/* FileInformationClass; =>FILE_BASIC_INFORMATION */
	if (GNOME_VFS_OK!=(errvfsresult=captive_NTSTATUS_to_GnomeVFSResult(err)))
		return errvfsresult;
	g_return_val_if_fail(NT_SUCCESS(err)==NT_SUCCESS(file_IoStatusBlock.Status),GNOME_VFS_ERROR_GENERIC);
	g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);
	errvfsresult=FileBasicInformationStruct_to_CaptiveFileInfoObject_in_place(captive_file_info_object,
			FileBasicInformationStruct);
	g_return_val_if_fail(errvfsresult==GNOME_VFS_OK,errvfsresult);

	/* query StandardInformation */
	FileStandardInformationStruct=(void *)QueryFile_buf;
	err=NtQueryInformationFile(
			captive_file_slave_object->file_Handle,	/* FileHandle */
			&file_IoStatusBlock,	/* IoStatusBlock */
			(gpointer)QueryFile_buf,	/* FileInformation */
			sizeof(QueryFile_buf),	/* Length */
			FileStandardInformation);	/* FileInformationClass; =>FILE_STANDARD_INFORMATION */
	if (GNOME_VFS_OK!=(errvfsresult=captive_NTSTATUS_to_GnomeVFSResult(err)))
		return errvfsresult;
	g_return_val_if_fail(NT_SUCCESS(err)==NT_SUCCESS(file_IoStatusBlock.Status),GNOME_VFS_ERROR_GENERIC);
	g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);
	errvfsresult=FileStandardInformationStruct_to_CaptiveFileInfoObject_in_place(captive_file_info_object,
			FileStandardInformationStruct);
	g_return_val_if_fail(errvfsresult==GNOME_VFS_OK,errvfsresult);

	*captive_file_info_object_return=captive_file_info_object;
	captive_leave();
	return GNOME_VFS_OK;
}


static GnomeVFSResult captive_file_slave_file_info_set
		(CaptiveFileObject *captive_file_object,CaptiveFileInfoObject *captive_file_info_object,GnomeVFSSetFileInfoMask mask)
{
CaptiveFileSlaveObject *captive_file_slave_object;

	g_return_val_if_fail(CAPTIVE_FILE_SLAVE_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(CAPTIVE_FILE_INFO_IS_OBJECT(captive_file_info_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=CAPTIVE_FILE_SLAVE_OBJECT(captive_file_object);

	g_return_val_if_fail(captive_file_slave_object->file_Handle!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	if (mask & GNOME_VFS_SET_FILE_INFO_NAME) {
gchar *name_dirname,*name_basename;
UNICODE_STRING *name_UnicodeString;
FILE_RENAME_INFORMATION *FileRenameInformation_structp;
gsize FileRenameInformation_struct_len;
IO_STATUS_BLOCK file_IoStatusBlock;
NTSTATUS err=STATUS_SUCCESS;	/* Prevent: ... might be used uninitialized in this function */
GnomeVFSResult errvfsresult;

		/* non-existing dirname assertion */
		name_dirname=g_path_get_dirname(captive_file_info_object->p.name);
		if (*name_dirname) {
			g_assert_not_reached();
			errvfsresult=GNOME_VFS_ERROR_BAD_PARAMETERS;
			goto err;
			}

		/* fully-matching basename assertion */
		name_basename=g_path_get_basename(captive_file_info_object->p.name);
		if (strcmp(name_basename,captive_file_info_object->p.name)) {
			g_assert_not_reached();
			errvfsresult=GNOME_VFS_ERROR_BAD_PARAMETERS;
			goto err_free_name_dirname;
			}

		name_UnicodeString=captive_utf8_to_UnicodeString_alloca(captive_file_info_object->p.name);
		FileRenameInformation_struct_len=sizeof(*FileRenameInformation_structp)
				+name_UnicodeString->MaximumLength;	/* ==Length+(0-terminator); is the terminator needed? */

		FileRenameInformation_structp=g_alloca(FileRenameInformation_struct_len);
		FileRenameInformation_structp->Replace=FALSE;
		FileRenameInformation_structp->RootDir=NULL;	/* we do 'simple rename' here */
		FileRenameInformation_structp->FileNameLength=name_UnicodeString->Length;
		memcpy(FileRenameInformation_structp->FileName,name_UnicodeString->Buffer,
				name_UnicodeString->MaximumLength);	/* ==Length+(0-terminator); is the terminator needed? */

		err=NtSetInformationFile(
				captive_file_slave_object->file_Handle,	/* FileHandle */
				&file_IoStatusBlock,	/* IoStatusBlock */
				FileRenameInformation_structp,	/* FileInformation */
				FileRenameInformation_struct_len,	/* Length */
				FileRenameInformation);	/* FileInformationClass */
		if (GNOME_VFS_OK!=(errvfsresult=captive_NTSTATUS_to_GnomeVFSResult(err)))
			goto err_free_name_basename;

err_free_name_basename:
		g_free(name_basename);
err_free_name_dirname:
		g_free(name_dirname);
err:

		if (errvfsresult!=GNOME_VFS_OK)
			return errvfsresult;
		g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);
		}

	if (mask & (GNOME_VFS_SET_FILE_INFO_PERMISSIONS | GNOME_VFS_SET_FILE_INFO_TIME)) {
char FileBasicInformation_buf[sizeof(FILE_BASIC_INFORMATION)+4];
FILE_BASIC_INFORMATION *FileBasicInformationp=(FILE_BASIC_INFORMATION *)FileBasicInformation_buf;
IO_STATUS_BLOCK file_IoStatusBlock;
NTSTATUS err;

		/* FIXME: Everywhere is is the declaration with 4xLARGE_INTEGER+1xULONG==36 bytes
		 * but the precompiled ext2fsd.sys wants 40 bytes - why?
		 * The same case applies to ntfs.sys of NT-5.1sp1.
		 */
		g_assert(sizeof(FILE_BASIC_INFORMATION)==36);
		g_assert(sizeof(FileBasicInformation_buf)==40);
		err=NtQueryInformationFile(
				captive_file_slave_object->file_Handle,	/* FileHandle */
				&file_IoStatusBlock,	/* IoStatusBlock */
				FileBasicInformation_buf,	/* FileInformation */
				sizeof(FileBasicInformation_buf),	/* Length */
				FileBasicInformation);	/* FileInformationClass */
		g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);

		if (mask & GNOME_VFS_SET_FILE_INFO_PERMISSIONS) {
			FileBasicInformationp->FileAttributes&=~FILE_ATTRIBUTE_READONLY;
			g_assert(captive_file_info_object->p.valid_fields & GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS);
			if (!(captive_file_info_object->p.permissions&0200))
				FileBasicInformationp->FileAttributes|=FILE_ATTRIBUTE_READONLY;
			}

		if (mask & GNOME_VFS_SET_FILE_INFO_TIME) {
			g_assert(captive_file_info_object->p.valid_fields & (0
					| GNOME_VFS_FILE_INFO_FIELDS_ATIME
					| GNOME_VFS_FILE_INFO_FIELDS_MTIME
					| GNOME_VFS_FILE_INFO_FIELDS_CTIME));
			if (captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_ATIME) {
struct timespec timespec;
GnomeVFSResult errvfsresult;

				timespec.tv_sec=captive_file_info_object->p.atime;
				timespec.tv_nsec=captive_file_info_object->atime_nsec;
				errvfsresult=timespec_to_Time(
						&FileBasicInformationp->LastAccessTime,	/* Time */
						&timespec);	/* timespec */
				g_return_val_if_fail(GNOME_VFS_OK==errvfsresult,errvfsresult);
				}
			if (captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_MTIME) {
struct timespec timespec;
GnomeVFSResult errvfsresult;

				timespec.tv_sec=captive_file_info_object->p.mtime;
				timespec.tv_nsec=captive_file_info_object->mtime_nsec;
				errvfsresult=timespec_to_Time(
						&FileBasicInformationp->LastWriteTime,	/* Time */
						&timespec);	/* timespec */
				g_return_val_if_fail(GNOME_VFS_OK==errvfsresult,errvfsresult);
				FileBasicInformationp->ChangeTime=FileBasicInformationp->LastWriteTime;
				}
			if (captive_file_info_object->p.valid_fields&GNOME_VFS_FILE_INFO_FIELDS_CTIME) {
struct timespec timespec;
GnomeVFSResult errvfsresult;

				timespec.tv_sec=captive_file_info_object->p.ctime;
				timespec.tv_nsec=captive_file_info_object->ctime_nsec;
				errvfsresult=timespec_to_Time(
						&FileBasicInformationp->CreationTime,	/* Time */
						&timespec);	/* timespec */
				g_return_val_if_fail(GNOME_VFS_OK==errvfsresult,errvfsresult);
				}
			}

		err=NtSetInformationFile(
				captive_file_slave_object->file_Handle,	/* FileHandle */
				&file_IoStatusBlock,	/* IoStatusBlock */
				FileBasicInformation_buf,	/* FileInformation */
				sizeof(FileBasicInformation_buf),	/* Length */
				FileBasicInformation);	/* FileInformationClass */
		g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);
		}

	if (mask & GNOME_VFS_SET_FILE_INFO_OWNER) {
		/* FIXME: g_assert(captive_file_info_object->p.valid_fields & ???); */
		/* owner ignored for W32 */
		}

	captive_leave();
	return GNOME_VFS_OK;
}


GnomeVFSResult captive_file_slave_truncate(CaptiveFileObject *captive_file_object,GnomeVFSFileSize file_size)
{
CaptiveFileSlaveObject *captive_file_slave_object;
NTSTATUS err;
FILE_END_OF_FILE_INFORMATION FileEndOfFileInformation_struct;
IO_STATUS_BLOCK file_IoStatusBlock;
GnomeVFSResult errvfsresult;

	g_return_val_if_fail(CAPTIVE_FILE_SLAVE_IS_OBJECT(captive_file_object),GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object=CAPTIVE_FILE_SLAVE_OBJECT(captive_file_object);

	g_return_val_if_fail(captive_file_slave_object->file_Handle!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	FileEndOfFileInformation_struct.EndOfFile.QuadPart=file_size;

	err=NtSetInformationFile(
			captive_file_slave_object->file_Handle,	/* FileHandle */
			&file_IoStatusBlock,	/* IoStatusBlock */
			&FileEndOfFileInformation_struct,	/* FileInformation */
			sizeof(FileEndOfFileInformation_struct),	/* Length */
			FileEndOfFileInformation);	/* FileInformationClass */
	if (GNOME_VFS_OK!=(errvfsresult=captive_NTSTATUS_to_GnomeVFSResult(err)))
		return errvfsresult;
	g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);

	captive_leave();
	return GNOME_VFS_OK;
}


GnomeVFSResult captive_file_slave_move
		(CaptiveFileObject *captive_file_object_old,const gchar *pathname_new,gboolean force_replace)
{
CaptiveFileSlaveObject *captive_file_slave_object_old;
NTSTATUS err;
FILE_RENAME_INFORMATION *FileRenameInformation_structp;
IO_STATUS_BLOCK file_IoStatusBlock;
GnomeVFSResult errvfsresult;
OBJECT_ATTRIBUTES pathname_new_ObjectAttributes;

	g_return_val_if_fail(CAPTIVE_FILE_SLAVE_IS_OBJECT(captive_file_object_old),GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(pathname_new!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	captive_file_slave_object_old=CAPTIVE_FILE_SLAVE_OBJECT(captive_file_object_old);

	g_return_val_if_fail(captive_file_slave_object_old->file_Handle!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	errvfsresult=captive_ObjectAttributes_init(pathname_new,&pathname_new_ObjectAttributes);
	g_return_val_if_fail(errvfsresult==GNOME_VFS_OK,errvfsresult);

	FileRenameInformation_structp=
			g_alloca(sizeof(*FileRenameInformation_structp)+pathname_new_ObjectAttributes.ObjectName->MaximumLength);
	FileRenameInformation_structp->Replace=force_replace;
	FileRenameInformation_structp->RootDir=0;	/* AFAIK never set by orig W32 */
	FileRenameInformation_structp->FileNameLength=pathname_new_ObjectAttributes.ObjectName->Length;
	memcpy(FileRenameInformation_structp->FileName,pathname_new_ObjectAttributes.ObjectName->Buffer,
			pathname_new_ObjectAttributes.ObjectName->MaximumLength);

	err=NtSetInformationFile(
			captive_file_slave_object_old->file_Handle,	/* FileHandle */
			&file_IoStatusBlock,	/* IoStatusBlock */
			FileRenameInformation_structp,	/* FileInformation */
			sizeof(*FileRenameInformation_structp)+pathname_new_ObjectAttributes.ObjectName->MaximumLength,	/* Length */
			FileRenameInformation);	/* FileInformationClass */
	if (GNOME_VFS_OK!=(errvfsresult=captive_NTSTATUS_to_GnomeVFSResult(err)))
		goto err_free_ObjectAttributes;
	g_return_val_if_fail(NT_SUCCESS(err),GNOME_VFS_ERROR_GENERIC);

	errvfsresult=GNOME_VFS_OK;

err_free_ObjectAttributes:
	g_free(pathname_new_ObjectAttributes.ObjectName);	/* left from captive_file_slave_uri_parent_init() */

	captive_leave();
	return errvfsresult;
}
