/* $Id: parent-connector.c,v 1.2 2003/09/12 19:42:35 short Exp $
 * fs object sandbox parents reconnections to their sandbox slave peers
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "parent-connector.h"	/* self */
#include <glib/gmessages.h>


/* g_object_get_data() 'key': */
#define CAPTIVE_PARENT_CONNECTOR_CORBA_OBJECTP             "captive_parent_connector-corba_objectp"	/* &CORBA_Object */
#define CAPTIVE_PARENT_CONNECTOR_CAPTIVE_VFS_PARENT_OBJECT "captive_parent_connector-captive_vfs_parent_object"	/* &CaptiveVfsParentObject */
#define CAPTIVE_PARENT_CONNECTOR_DIRTY                     "captive_parent_connector-dirty"	/* GINT_TO_POINTER(0/1) */
#define CAPTIVE_PARENT_CONNECTOR_BROKEN                    "captive_parent_connector-broken"	/* GINT_TO_POINTER(0/1) */
#define CAPTIVE_PARENT_CONNECTOR_SELF_REFS                 "captive_parent_connector-self_refs"	/* GINT_TO_POINTER(count) */


/*
 * vfs signal <detach>
 * vfs signal <cease>
 * vfs signal <abort>
 * 
 * connector state vfs          yes/no
 * connector state corba_object yes/no
 * connector state dirty        yes/no
 * 
 *                 vfs     handle  dirty   oper.   detach  cease   abort
 * disconnected    N       N       N       opn_*   discon. discon. discon.
 * broken          N       N       Y       broken  broken  broken  broken
 * N/A             N       Y       N
 * N/A             N       Y       Y
 * closed_clean    Y       N       N       X       X       discon. discon.
 * closed_dirty    Y       N       Y       X       X       discon. broken
 * opened_clean    Y       Y       N       opn_*   cls_cln X       discon.
 * opened_dirty    Y       Y       Y       opn_drt cls_drt X       broken
 * 
 * init->disconnected
 * 
 * <detach>[VFS shutdown]<cease>
 * <detach>[failure]<abort>
 * [failure]<abort>
 * 
 * 
 * sandbox master GObject hierarchy (horizontal=dervived):
 *         DirectoryParent----------------------------------->Directory->GObject
 *                        \->ParentConnector->GTypeInterface   |
 *                                  | [ref.]                   | [reference]
 *                                 VfsParent----------------->Vfs->GObject
 * 
 * sandbox master GObject reconnections signalling:
 *               VfsParent -> ParentConnector
 *         ParentConnector -> DirectoryParent
 * sandbox master reconnection requests:
 *         DirectoryParent -> ParentConnector
 *         ParentConnector -> VfsParent
 * 
 * CORBA/ORBit connections:
 *         sandbox master                   |  sandbox slave (restartable)
 *         ---------------------------------|-----------------------------------
 *         DirectoryParent--Directory_stub <-> Directory_servant--DirectorySlave
 *               VfsParent--------Vfs_stub <->       Vfs_servant--VfsSlave
 * 
 * sandbox slave GObject hierarchy (horizontal=dervived):
 *         DirectorySlave->Directory->GObject
 *                          | [reference]
 *               VfsSlave->Vfs->GObject
 */


static GnomeVFSResult captive_parent_connector_handler_open(CaptiveParentConnector *captive_parent_connector);
static GnomeVFSResult captive_parent_connector_handler_close(CaptiveParentConnector *captive_parent_connector);


static void captive_parent_connector_iface_init(CaptiveParentConnectorIface *captive_parent_connector_iface)
{
	captive_parent_connector_iface->open=captive_parent_connector_handler_open;
	captive_parent_connector_iface->close=captive_parent_connector_handler_close;
}


GType captive_parent_connector_get_type(void)
{
static GType captive_parent_connector_type=0;

	if (!captive_parent_connector_type) {
static const GTypeInfo captive_parent_connector_info={
				sizeof(CaptiveParentConnectorIface),
				(GBaseInitFunc)captive_parent_connector_iface_init,	/* base_init */
				NULL,	/* base_finalize */
				NULL,	/* iface_init */
				NULL,	/* iface_finalize */
				NULL,	/* iface_data */
				0,	/* instance_size */
				0,	/* n_preallocs */
				NULL,	/* instance_init */
				};

		captive_parent_connector_type=g_type_register_static(G_TYPE_INTERFACE,
				"CaptiveParentConnector",&captive_parent_connector_info,0);
		}

	return captive_parent_connector_type;
}

static void captive_parent_connector_vfs_signals_disconnect(CaptiveParentConnector *captive_parent_connector);

static void captive_parent_connector_vfs_detach(CaptiveVfsParentObject *captive_vfs_parent_object,
		CaptiveParentConnector *captive_parent_connector)
{
	g_return_if_fail(CAPTIVE_VFS_PARENT_IS_OBJECT(captive_vfs_parent_object));
	g_return_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector));
	g_return_if_fail(NULL==captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector)
			|| captive_vfs_parent_object==captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector));

	/**/ if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED))
		/* NOP */;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_BROKEN))
		/* NOP */;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED))
		/* NOP */;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED)) {
		(*CAPTIVE_PARENT_CONNECTOR_GET_IFACE(captive_parent_connector)->close)(captive_parent_connector);	/* errors ignored */
		g_assert(captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED));
		}
	else g_assert_not_reached();
}

static void captive_parent_connector_vfs_cease(CaptiveVfsParentObject *captive_vfs_parent_object,
		CaptiveParentConnector *captive_parent_connector)
{
	g_return_if_fail(CAPTIVE_VFS_PARENT_IS_OBJECT(captive_vfs_parent_object));
	g_return_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector));
	g_return_if_fail(NULL==captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector)
			|| captive_vfs_parent_object==captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector));

	/**/ if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED))
		/* NOP */;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_BROKEN))
		/* NOP */;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED)) {
		g_object_set_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_DIRTY,
				GINT_TO_POINTER(FALSE));
		}
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED))
		g_assert_not_reached();
	else g_assert_not_reached();

	captive_parent_connector_vfs_signals_disconnect(captive_parent_connector);
}

static void captive_parent_connector_vfs_abort(CaptiveVfsParentObject *captive_vfs_parent_object,
		CaptiveParentConnector *captive_parent_connector)
{
	g_return_if_fail(CAPTIVE_VFS_PARENT_IS_OBJECT(captive_vfs_parent_object));
	g_return_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector));
	g_return_if_fail(NULL==captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector)
			|| captive_vfs_parent_object==captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector));

	if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED_DIRTY_OR_OPENED_DIRTY))
		g_warning(_("Filesystem crash broke dirty object: %s"),
				(*CAPTIVE_PARENT_CONNECTOR_GET_IFACE(captive_parent_connector)->get_pathname)(captive_parent_connector));

	/**/ if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED))
		/* NOP */;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_BROKEN))
		/* NOP */;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED_CLEAN))
		/* NOP */;	/* -> DISCONNECTED */
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED_DIRTY)) {
		/* -> BROKEN */
		/* Ensure to never become VFS-capable Connector from BROKEN state. */
		g_object_set_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_BROKEN,
				GINT_TO_POINTER(TRUE));
		}
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED)) {
		(*CAPTIVE_PARENT_CONNECTOR_GET_IFACE(captive_parent_connector)->close)(captive_parent_connector);	/* errors ignored */
		g_assert(captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED));
		}
	else g_assert_not_reached();

	captive_parent_connector_vfs_signals_disconnect(captive_parent_connector);
}

static void captive_parent_connector_vfs_signals_connect(CaptiveParentConnector *captive_parent_connector)
{
CaptiveVfsParentObject *captive_vfs_parent_object;

	g_return_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector));

	captive_vfs_parent_object=captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector);

	/* Prevent multiple equivalent signal handlers. */
	captive_parent_connector_vfs_signals_disconnect(captive_parent_connector);

	g_signal_connect(captive_vfs_parent_object,"detach",
			G_CALLBACK(captive_parent_connector_vfs_detach),captive_parent_connector);	/* c_handler,data */
	g_signal_connect(captive_vfs_parent_object,"cease",
			G_CALLBACK(captive_parent_connector_vfs_cease),captive_parent_connector);	/* c_handler,data */
	g_signal_connect(captive_vfs_parent_object,"abort",
			G_CALLBACK(captive_parent_connector_vfs_abort),captive_parent_connector);	/* c_handler,data */
}

static void captive_parent_connector_vfs_signals_disconnect(CaptiveParentConnector *captive_parent_connector)
{
CaptiveVfsParentObject *captive_vfs_parent_object;
gint self_refs;

	g_return_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector));

	captive_vfs_parent_object=captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector);

	g_signal_handlers_disconnect_by_func(captive_vfs_parent_object,
			G_CALLBACK(captive_parent_connector_vfs_detach),captive_parent_connector);	/* func,data */
	g_signal_handlers_disconnect_by_func(captive_vfs_parent_object,
			G_CALLBACK(captive_parent_connector_vfs_cease),captive_parent_connector);	/* func,data */
	g_signal_handlers_disconnect_by_func(captive_vfs_parent_object,
			G_CALLBACK(captive_parent_connector_vfs_abort),captive_parent_connector);	/* func,data */

	/* Do not 'while'-cycle the loop by 1-decreasing as we cannot g_object_get_data()
	 * on fully unreffed GObject.
	 */
	if ((self_refs=GPOINTER_TO_INT(
			g_object_get_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_SELF_REFS)))) {
		g_assert(self_refs>0);
		g_object_set_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_SELF_REFS,
				GINT_TO_POINTER(0));
		while (self_refs--) {
			/* Undo captive_directory_close() / captive_file_close(): */
			g_object_ref(captive_vfs_parent_object);
			/* Cause g_object_last_unref(): */
			g_object_unref(captive_parent_connector);
			}
		}
}


void captive_parent_connector_init(CaptiveParentConnector *captive_parent_connector,
		CORBA_Object *corba_objectp,CaptiveVfsParentObject *captive_vfs_parent_object)
{
	g_return_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector));
	g_return_if_fail(corba_objectp!=NULL);
	g_return_if_fail(/* CORBA_Object_is_nil() */ *corba_objectp==CORBA_OBJECT_NIL);
	g_return_if_fail(CAPTIVE_VFS_PARENT_IS_OBJECT(captive_vfs_parent_object));

	g_object_set_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_CORBA_OBJECTP,
			corba_objectp);
	g_object_set_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_CAPTIVE_VFS_PARENT_OBJECT,
			captive_vfs_parent_object);
	g_object_set_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_DIRTY,
			GINT_TO_POINTER(FALSE));

	g_assert(
			captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED_OR_CLOSED_CLEAN));
}


/* Abort dispose()? */
gboolean captive_parent_connector_dispose(CaptiveParentConnector *captive_parent_connector)
{
	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),FALSE);

	if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED_DIRTY)) {
CaptiveVfsParentObject *captive_vfs_parent_object;

		captive_vfs_parent_object=captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector);

		/* Cancel g_object_last_unref(): */
		g_object_ref(captive_parent_connector);
		/* Simulate captive_directory_close() / captive_file_close(): */
		g_object_unref(captive_vfs_parent_object);
		g_object_set_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_SELF_REFS,
				GINT_TO_POINTER(1+GPOINTER_TO_INT(
						g_object_get_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_SELF_REFS))));
		return TRUE;	/* abort dispose() */
		}

	captive_parent_connector_vfs_signals_disconnect(captive_parent_connector);
	return FALSE;	/* continue dispose() */
}


void captive_parent_connector_finalize(CaptiveParentConnector *captive_parent_connector)
{
	g_return_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector));

	g_assert(captive_parent_connector_is_state(
			captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED_OR_BROKEN_OR_CLOSED));
	/* Should have been catched by captive_parent_connector_dispose(): */
	g_assert(!captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED_DIRTY));

	captive_parent_connector_vfs_signals_disconnect(captive_parent_connector);
}


GnomeVFSResult captive_parent_connector_connect(CaptiveParentConnector *captive_parent_connector)
{
GnomeVFSResult r;

	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),GNOME_VFS_ERROR_BAD_PARAMETERS);

	if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED)) {
		if (GNOME_VFS_OK!=(r=captive_vfs_parent_object_connect(
				captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector))))
			return r;
		g_assert(captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED_CLEAN));
		}

	/**/ if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED))
		g_assert_not_reached();
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_BROKEN))
		return GNOME_VFS_ERROR_SERVICE_NOT_AVAILABLE;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED))
		return GNOME_VFS_OK;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED))
		return GNOME_VFS_OK;
	else g_assert_not_reached();

	/* NOTREACHED */
	return GNOME_VFS_ERROR_INTERNAL;
}


GnomeVFSResult captive_parent_connector_open(CaptiveParentConnector *captive_parent_connector)
{
GnomeVFSResult r;

	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),GNOME_VFS_ERROR_BAD_PARAMETERS);

	if (GNOME_VFS_OK!=(r=captive_parent_connector_connect(captive_parent_connector)))
		return r;

	if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED)) {
		if (GNOME_VFS_OK!=(r=(*CAPTIVE_PARENT_CONNECTOR_GET_IFACE(captive_parent_connector)->open)(captive_parent_connector)))
			return r;
		g_assert(captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED));
		}

	/**/ if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED))
		g_assert_not_reached();
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_BROKEN))
		g_assert_not_reached();
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED))
		g_assert_not_reached();
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED))
		return GNOME_VFS_OK;
	else g_assert_not_reached();

	/* NOTREACHED */
	return GNOME_VFS_ERROR_INTERNAL;
}


GnomeVFSResult captive_parent_connector_close(CaptiveParentConnector *captive_parent_connector)
{
GnomeVFSResult r;

	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),GNOME_VFS_ERROR_BAD_PARAMETERS);

	if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED)) {
		if (GNOME_VFS_OK!=(r=(*CAPTIVE_PARENT_CONNECTOR_GET_IFACE(captive_parent_connector)->close)(captive_parent_connector)))
			return r;
		g_assert(captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED));
		}

	/**/ if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED))
		return GNOME_VFS_OK;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_BROKEN))
		return GNOME_VFS_OK;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED))
		return GNOME_VFS_OK;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED))
		g_assert_not_reached();
	else g_assert_not_reached();

	/* NOTREACHED */
	return GNOME_VFS_ERROR_INTERNAL;
}


void captive_parent_connector_set_dirty(CaptiveParentConnector *captive_parent_connector)
{
	g_return_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector));

	g_return_if_fail(captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED));

	g_object_set_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_DIRTY,
			GINT_TO_POINTER(TRUE));
}


static GnomeVFSResult captive_parent_connector_handler_open(CaptiveParentConnector *captive_parent_connector)
{
	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),GNOME_VFS_ERROR_BAD_PARAMETERS);

	/* Handler should have 'opened' us. */

	/**/ if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED))
		g_assert_not_reached();
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_BROKEN))
		g_assert_not_reached();
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED))
		g_assert_not_reached();	/* Parent handler failed. */
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED)) {
		captive_parent_connector_vfs_signals_connect(captive_parent_connector);
		return GNOME_VFS_OK;
		}
	else g_assert_not_reached();

	/* NOTREACHED */
	return GNOME_VFS_ERROR_INTERNAL;
}


static GnomeVFSResult captive_parent_connector_handler_close(CaptiveParentConnector *captive_parent_connector)
{
	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),GNOME_VFS_ERROR_BAD_PARAMETERS);

	/* Handler should have 'closed' us. */

	/**/ if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_DISCONNECTED))
		return GNOME_VFS_OK;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_BROKEN))
		return GNOME_VFS_OK;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_CLOSED))
		return GNOME_VFS_OK;
	else if (captive_parent_connector_is_state(captive_parent_connector,CAPTIVE_PARENT_CONNECTOR_FLAGS_OPENED))
		g_assert_not_reached();
	else g_assert_not_reached();

	/* NOTREACHED */
	return GNOME_VFS_ERROR_INTERNAL;
}


CaptiveVfsParentObject *captive_parent_connector_get_captive_vfs_parent_object
		(CaptiveParentConnector *captive_parent_connector)
{
CaptiveVfsParentObject *r;

	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),NULL);

	r=g_object_get_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_CAPTIVE_VFS_PARENT_OBJECT);
	g_assert(CAPTIVE_VFS_PARENT_IS_OBJECT(r));

	return r;
}


CORBA_Object captive_parent_connector_get_corba_object(CaptiveParentConnector *captive_parent_connector)
{
CORBA_Object *rp;

	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),NULL);

	rp=g_object_get_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_CORBA_OBJECTP);
	g_assert(rp!=NULL);
	/* '*rp' may be NULL */

	return *rp;
}


gboolean captive_parent_connector_get_dirty(CaptiveParentConnector *captive_parent_connector)
{
gpointer r_gpointer;
gint r_gint;

	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),FALSE);

	r_gpointer=g_object_get_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_DIRTY);
	r_gint=GPOINTER_TO_INT(r_gpointer);
	g_assert(r_gint==TRUE || r_gint==FALSE);

	return r_gint;
}


gboolean captive_parent_connector_is_state(CaptiveParentConnector *captive_parent_connector,
		CaptiveParentConnectorFlagWant vfs_want,
		CaptiveParentConnectorFlagWant corba_object_want,CaptiveParentConnectorFlagWant dirty_want)
{
	g_return_val_if_fail(CAPTIVE_IS_PARENT_CONNECTOR(captive_parent_connector),FALSE);

	if (vfs_want!=CAPTIVE_PARENT_CONNECTOR_FLAG_WANT_ANY) {
CaptiveVfsParentObject *captive_vfs_parent_object;

		captive_vfs_parent_object=captive_parent_connector_get_captive_vfs_parent_object(captive_parent_connector);
		if ((vfs_want==CAPTIVE_PARENT_CONNECTOR_FLAG_WANT_ON)!=
				(!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_BROKEN))
						&& /* !CORBA_Object_is_nil() */ captive_vfs_parent_object->corba_Vfs_object!=CORBA_OBJECT_NIL))
			return FALSE;
		}
	if (corba_object_want!=CAPTIVE_PARENT_CONNECTOR_FLAG_WANT_ANY) {
CORBA_Object corba_object;

		corba_object=captive_parent_connector_get_corba_object(captive_parent_connector);
		g_assert(!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_BROKEN))
				|| corba_object==CORBA_OBJECT_NIL);
		if ((corba_object_want==CAPTIVE_PARENT_CONNECTOR_FLAG_WANT_ON)!=
				(/* !CORBA_Object_is_nil() */ corba_object!=CORBA_OBJECT_NIL))
			return FALSE;
		}
	if (dirty_want!=CAPTIVE_PARENT_CONNECTOR_FLAG_WANT_ANY) {
gboolean dirty;

		dirty=captive_parent_connector_get_dirty(captive_parent_connector);
		g_assert(!GPOINTER_TO_INT(g_object_get_data(G_OBJECT(captive_parent_connector),CAPTIVE_PARENT_CONNECTOR_BROKEN))
				|| dirty==TRUE);
		if ((dirty_want==CAPTIVE_PARENT_CONNECTOR_FLAG_WANT_ON)!=dirty)
			return FALSE;
		}

	return TRUE;
}
