/* $Id: mdl.c,v 1.6 2003/03/27 11:33:07 short Exp $
 * reactos MDL emulation of libcaptive
 * Copyright (C) 2002 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "reactos/ddk/mmfuncs.h"	/* self */
#include <glib/gmessages.h>


/**
 * MmProbeAndLockPages:
 * @Mdl: Pointer to the #MDL specification structure to probe and lock.
 * %NULL value is forbidden.
 * @AccessMode: Access at which to lock the pages. %KernelMode required by libcaptive.
 * @Operation: One of %IoReadAccess, %IoWriteAccess or %IoModifyAccess.
 *
 * Probles the specified @AccessMode and locks those pages specified by @Mdl to memory
 * with the desired @AccessMode.
 * libcaptive does NOP here as it just sets the %MDL_PAGES_LOCKED flag.
 */
VOID MmProbeAndLockPages(PMDL Mdl,KPROCESSOR_MODE AccessMode,LOCK_OPERATION Operation)
{
	g_return_if_fail(Mdl!=NULL);
	g_return_if_fail(AccessMode==KernelMode /* || AccessMode==UserMode */);

	Mdl->MdlFlags|=MDL_PAGES_LOCKED;
}


/**
 * MmUnlockPages:
 * @Mdl: Pointer to the #MDL specification structure to unlock.
 * %NULL value is forbidden.
 *
 * Unlock the pages specified by @Mdl from memory.
 * libcaptive does NOP here as it just clears the %MDL_PAGES_LOCKED flag.
 */
VOID MmUnlockPages(PMDL Mdl)
{
	g_return_if_fail(Mdl!=NULL);

	Mdl->MdlFlags&=~MDL_PAGES_LOCKED;
}


/**
 * MmMapLockedPages:
 * @Mdl: Pointer to the #MDL specification structure to map.
 * %NULL value is forbidden.
 * @AccessMode: Access at which to lock the pages. %KernelMode required by libcaptive.
 *
 * Leaves the pages at their original location as they are never moved anyway
 * by libcaptive.
 *
 * Returns: Address of the mapped pages base.
 * This offset does not neet to be %PAGE_SIZE aligned.
 */
PVOID MmMapLockedPages(PMDL Mdl,KPROCESSOR_MODE AccessMode)
{
	g_return_val_if_fail(Mdl!=NULL,NULL);
	g_assert(Mdl->StartVa!=NULL);
	g_return_val_if_fail(AccessMode==KernelMode /* || AccessMode==UserMode */ ,NULL);

	if (Mdl->MdlFlags&MDL_MAPPED_TO_SYSTEM_VA) {
		/* already mapped */
		g_assert(Mdl->MappedSystemVa==Mdl->StartVa+Mdl->ByteOffset);
		}
	else {
		/* new mapping */
		Mdl->MappedSystemVa=Mdl->StartVa+Mdl->ByteOffset;
		Mdl->MdlFlags|=MDL_MAPPED_TO_SYSTEM_VA;
		}

	return Mdl->MappedSystemVa;
}


/**
 * MmUnmapLockedPages:
 * @BaseAddress: Address of the mapped pages base.
 * This offset does not neet to be %PAGE_SIZE aligned.
 * @Mdl: Pointer to the #MDL specification structure to unmap.
 *
 * Declares the specified @Mdl pages as unmapped. @BaseAddress
 * is required to be previously returned by MmMapLockedPages().
 * You must not call this function twice without MmMapLockedPages() between.
 */
VOID MmUnmapLockedPages(PVOID BaseAddress,PMDL Mdl)
{
	g_return_if_fail(BaseAddress!=NULL);
	g_return_if_fail(Mdl!=NULL);
	g_return_if_fail(BaseAddress==Mdl->MappedSystemVa);

	/* No mapping is done for pages from MmBuildMdlForNonPagedPool(). */
	if (Mdl->MdlFlags&MDL_SOURCE_IS_NONPAGED_POOL) {
		g_return_if_fail(!(Mdl->MdlFlags&MDL_MAPPED_TO_SYSTEM_VA));
		}
	else {
		g_return_if_fail(Mdl->MdlFlags&MDL_MAPPED_TO_SYSTEM_VA);
		Mdl->MdlFlags&=~MDL_MAPPED_TO_SYSTEM_VA;
		}
	Mdl->MappedSystemVa=NULL;
}

/**
 * MmMapLockedPagesSpecifyCache:
 * @Mdl: Pointer to the #MDL specification structure to map.
 * %NULL value is forbidden.
 * @AccessMode: Access at which to lock the pages. %KernelMode required by libcaptive.
 * @CacheType: Suggested caching type. %MmNonCached, %MmCached or %MmWriteCombined required by libcaptive.
 * @BaseAddress: Required block base address if @AccessMode is %UserMode. Ignored by libcaptive.
 * @BugCheckOnFailure: Whether to call KeBugCheck() instead of returning %NULL. Ignored by libcaptive.
 * @Priority: Suggested pages priority. %LowPagePriority, %NormalPagePriority or %HighPagePriority required by libcaptive.
 *
 * Leaves the pages at their original location as they are never moved anyway
 * by libcaptive. This call is just a wrapper around MmMapLockedPages() in libcaptive.
 *
 * Returns: Address of the mapped pages base.
 * This offset does not neet to be %PAGE_SIZE aligned.
 */
PVOID MmMapLockedPagesSpecifyCache(IN PMDL Mdl,
		IN KPROCESSOR_MODE AccessMode,IN MEMORY_CACHING_TYPE CacheType,IN PVOID BaseAddress,
		IN ULONG BugCheckOnFailure,IN MM_PAGE_PRIORITY Priority)
{
	g_return_val_if_fail(Mdl!=NULL,NULL);
	g_return_val_if_fail(AccessMode==KernelMode /* || AccessMode==UserMode */ ,NULL);
	g_return_val_if_fail(0
					|| CacheType==MmNonCached
					|| CacheType==MmCached
					|| CacheType==MmWriteCombined,
			NULL);
	g_return_val_if_fail(0
					|| Priority==LowPagePriority
					|| Priority==NormalPagePriority
					|| Priority==HighPagePriority,
			NULL);

	return MmMapLockedPages(Mdl,AccessMode);
}
