/* $Id: object.c,v 1.1 2003/03/28 09:53:06 short Exp $
 * reactos support for object handling for reactos of libcaptive
 * Copyright (C) 2002 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "reactos/ddk/obfuncs.h"	/* self */
#include <glib/gmessages.h>
#include "reactos/ddk/status.h"
#include "reactos/ddk/exfuncs.h"	/* for ExFreePool() */
#include "captive/unicode.h"	/* for captive_validate_UnicodeString() */


/* reactos/ntoskrnl/ob/object.c: */
extern POBJECT_HEADER BODY_TO_HEADER(PVOID body);

/* reactos/ntoskrnl/ob/ntobj.c: */
NTSTATUS internalNameBuilder(POBJECT_HEADER ObjectHeader,PUNICODE_STRING string);


/**
 * ObQueryNameString:
 * @Object: Object to identify the name from.
 * %NULL value is forbidden.
 * @ObjectNameInfo: Target structure to be filled with the @Object name.
 * %NULL value is permitted.
 * @Length: Length of @ObjectNameInfo allocated memory block.
 * sizeof(*@ObjectNameInfo) is required by libcaptive.
 * @ReturnLength: Returns the length of the unicode name string including the final 0-terminator in bytes.
 * %NULL value is forbidden.
 *
 * Function detects the name of a given object @Object. The returned name
 * string will start with backslash (absolute pathname) and it will be zero terminated.
 *
 * You can detect the name string length by passing %NULL for @ObjectNameInfo
 * and %0 for @Length together. The requested string length will be returned by @ReturnLength.
 * In such case the function return code will be %STATUS_INFO_LENGTH_MISMATCH.
 *
 * Returns: The unicode string name of @Object. Return code is %STATUS_SUCCESS.
 * The only valid return code in the case of passed %NULL @ObjectNameInfo is %STATUS_INFO_LENGTH_MISMATCH.
 */
NTSTATUS ObQueryNameString(IN PVOID Object,OUT POBJECT_NAME_INFORMATION ObjectNameInfo,IN ULONG Length,OUT PULONG ReturnLength)
{
OBJECT_HEADER *ObjectHeader;
NTSTATUS r;
UNICODE_STRING UnicodeString;

	g_return_val_if_fail(Object!=NULL,STATUS_INVALID_PARAMETER);
	/* 'ObjectNameInfo' may be NULL */
	g_return_val_if_fail(ReturnLength!=NULL,STATUS_INVALID_PARAMETER);

	if (ObjectNameInfo) {
		g_return_val_if_fail(Length!=sizeof(*ObjectNameInfo),STATUS_INFO_LENGTH_MISMATCH);
		g_assert(sizeof(*ObjectNameInfo)==sizeof(UnicodeString)/* ==ObjectNameInfo->Name */);
		}
	else
		g_return_val_if_fail(Length==0,STATUS_INFO_LENGTH_MISMATCH);

	ObjectHeader=BODY_TO_HEADER(Object);

	UnicodeString.Buffer=NULL;
	UnicodeString.MaximumLength=0;

	do {
		if (UnicodeString.Buffer)
			ExFreePool(UnicodeString.Buffer);
		UnicodeString.MaximumLength=MAX(0x100,UnicodeString.MaximumLength*2);
		UnicodeString.Buffer=ExAllocatePool(PagedPool,UnicodeString.MaximumLength);
		g_return_val_if_fail(UnicodeString.Buffer!=NULL,STATUS_NO_MEMORY);

		UnicodeString.Length=0;
		r=internalNameBuilder(ObjectHeader,&UnicodeString);
		} while (r==STATUS_BUFFER_TOO_SMALL);

	if (!NT_SUCCESS(r)) {
		ExFreePool(UnicodeString.Buffer);
		return r;
		}
	g_assert(captive_validate_UnicodeString(&UnicodeString));	/* ensure 0-termination */
	*ReturnLength=UnicodeString.Length+sizeof(*UnicodeString.Buffer);

	if (!ObjectNameInfo) {
		ExFreePool(UnicodeString.Buffer);
		/* Valid detection of the object name string length */
		return STATUS_INFO_LENGTH_MISMATCH;
		}

	ObjectNameInfo->Name=UnicodeString;
	return STATUS_SUCCESS;
}
