/* $Id: file.c,v 1.3 2003/09/13 06:50:58 short Exp $
 * Internal (rtl/) file handling utilities of libcaptive
 * Copyright (C) 2002 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "captive/rtl-file.h"	/* self */
#include <glib/gtypes.h>
#include <glib/gmessages.h>
#include <glib/ghash.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <glib/gstring.h>
#include <errno.h>
#include <glib/galloca.h>
#include "reactos/internal/mm.h"	/* for PAGE_SIZE */
#include <glib/gmem.h>


/* map: address -> GSIZE_TO_POINTER(length) */
static GHashTable *captive_rtl_file_mmap_hash;

/* initialize 'captive_rtl_file_mmap_hash' */
static void captive_rtl_file_mmap_hash_init(void)
{
	if (captive_rtl_file_mmap_hash)
		return;
	captive_rtl_file_mmap_hash=g_hash_table_new(g_direct_hash,g_direct_equal);
}

/**
 * captive_rtl_file_mmap:
 * @lenp: returns the file length if successful. %NULL pointer permitted.
 * @path: File pathname to open(2).
 * @open_flags: open(2) parameter #flags such as %O_RDONLY.
 * @mmap_prot: mmap(2) parameter #prot such as %PROT_READ.
 * @mmap_flags: mmap(2) parameter #flags such as %MAP_SHARED.
 *
 * mmap(2) the whole file into memory.
 *
 * Returns: address base with mmap(2)ed file (and @lenp filled) or #NULL if error.
 */
gpointer captive_rtl_file_mmap(size_t *lenp,const gchar *path,int open_flags,int mmap_prot,int mmap_flags)
{
gpointer r;
int fd;
size_t len;
off_t lenoff;

	captive_rtl_file_mmap_hash_init();

	/* FIXME: use g_filename_from_utf8() on 'path' */
	fd=open(path,open_flags);
	if (fd==-1) {
		g_error("captive_rtl_file_mmap(\"%s\"): open: %m",path);
		goto err;
		}

	lenoff=lseek(fd,0,SEEK_END);
	if (lenoff==(off_t)-1) {
		g_assert_not_reached();
		goto err_close;
		}
	len=(size_t)lenoff;
	if (lenp)
		*lenp=len;

	r=mmap(
			0,	/* start */
			len,	/* len */
			mmap_prot,
			mmap_flags,
			fd,
			0	/* offset */
			);
	if (r==MAP_FAILED || r==NULL) {
		g_assert_not_reached();
		goto err_close;
		}

	if (close(fd)) {
		g_assert_not_reached();
		goto err_munmap;
		}

	/* should make no difference with g_hash_table_replace()
	 * as we are using (g_direct_hash,g_direct_equal)
	 */
	g_hash_table_insert(captive_rtl_file_mmap_hash,
			r,GSIZE_TO_POINTER(len));	/* key,value */

	return r;

err_munmap:
	/* errors ignored */
	munmap(r,len);
err_close:
	/* errors ignored */
	close(fd);
err:
	g_return_val_if_reached(NULL);
}

/**
 * captive_rtl_file_munmap:
 * @base: file base address returned by captive_rtl_file_mmap(). %NULL pointer forbidden.
 *
 * munmap(2) the whole file and destroy all its resources.
 */
void captive_rtl_file_munmap(gpointer base)
{
gpointer len_ptr;	/* GSIZE_TO_POINTER(off_t len) */
gboolean found;

	g_return_if_fail(base!=NULL);

	captive_rtl_file_mmap_hash_init();

	/* report if 'base' not found */
	found=g_hash_table_lookup_extended(captive_rtl_file_mmap_hash,
			base,	/* lookup_key */
			NULL,	/* orig_key */
			&len_ptr);	/* value */
	g_return_if_fail(found==TRUE);

	/* remove 'base' from our database */
	found=g_hash_table_remove(captive_rtl_file_mmap_hash,
			base);	/* key */
	g_return_if_fail(found==TRUE);	/* just a warning would be also possible */

	/* munmap() has no return value */
	munmap(base,GPOINTER_TO_SIZE(len_ptr));
}


/**
 * captive_rtl_file_read:
 * @fd: file-descriptor to read data from.
 * @bufsizep: Returns the size of resulting data.
 * %NULL pointer is permitted.
 *
 * Reads the whole file into memory.
 *
 * Returns: address base of the memory being filled with file contents.
 * Free the area by g_free() if no longer used.
 */
gpointer captive_rtl_file_read(gint fd,gsize *bufsizep)
{
off_t lenoff,gotoff;
GString *r_GString;
void *buf;
ssize_t readgot;

	errno=0;
	lenoff=lseek(fd,0,SEEK_END);
	switch (lenoff) {
		case (off_t)-1:
			if (errno!=ESPIPE)
				g_assert_not_reached();
			lenoff=PAGE_SIZE;
			break;
		case 0:
			return NULL;
		default:
			gotoff=lseek(fd,0,SEEK_SET);
			g_assert(gotoff==0);
		}
	r_GString=g_string_sized_new(lenoff);
	buf=g_malloc(lenoff);	/* do not use g_alloca() - the buffer may be big! */
	while ((readgot=read(fd,buf,lenoff))>0) {
		g_assert(readgot<=lenoff);
		r_GString=g_string_append_len(r_GString,buf,readgot);
		}
	g_assert(readgot==0);	 /* EOF */
	g_free(buf);

	if (bufsizep)
		*bufsizep=r_GString->len;
	return g_string_free(r_GString,
			FALSE);	/* free_segment */
}
