/* $Id: time.c,v 1.3 2006/01/16 18:19:15 lace Exp $
 * reactos time user/kernel memory handling functions emulation of libcaptive
 * Copyright (C) 2005 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "captive/rtl-time.h"	/* self */
#include <glib/gmessages.h>
#include "reactos/ntos/rtl.h"	/* For RtlSecondsSince1970ToTime() */


/* It MUST match those of: reactos/ntoskrnl/rtl/time.c */
#define TICKSPERSEC        10000000
#define TICKSTO1970         0x019db1ded53e8000LL


/** timespec_to_Time:
 * @Time: Target %PLARGE_INTEGER type target MS-Windows timestamp.
 * %NULL value is forbidden.
 * @timespec: Source %struct %timespec type UNIX timestamp.
 * %NULL value is forbidden.
 *
 * Converts the timestamp format including the sub-second part not handled by
 * just ReactOS RtlSecondsSince1970ToTime().
 *
 * Returns: %GNOME_VFS_OK if everything got converted OK.
 */
GnomeVFSResult timespec_to_Time(PLARGE_INTEGER Time,const struct timespec *timespec)
{
	g_return_val_if_fail(Time!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(timespec!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(timespec->tv_nsec>=0,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(timespec->tv_nsec<1000000000,GNOME_VFS_ERROR_BAD_PARAMETERS);

	RtlSecondsSince1970ToTime(timespec->tv_sec,Time);

	/* #define TICKSPERSEC 10000000 */
#if TICKSPERSEC<1000000000 && !(1000000000%TICKSPERSEC)
	Time->QuadPart+=timespec->tv_nsec / (1000000000/TICKSPERSEC);
#else
#error "Unsupported: TICKSPERSEC"
#endif

	return GNOME_VFS_OK;
}

/** Time_to_timespec:
 * @timespec: Target %struct %timespec type UNIX timestamp.
 * %NULL value is forbidden.
 * @Time: Source %PLARGE_INTEGER type target MS-Windows timestamp.
 * %NULL value is forbidden.
 *
 * Converts the timestamp format including the sub-second part not handled by
 * just ReactOS RtlTimeToSecondsSince1970().
 *
 * Returns: %GNOME_VFS_OK if everything got converted OK.
 */
GnomeVFSResult Time_to_timespec(struct timespec *timespec,const PLARGE_INTEGER Time)
{
ULONG SecondsSince1970;
BOOLEAN errBOOLEAN;

	g_return_val_if_fail(timespec!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);
	g_return_val_if_fail(Time!=NULL,GNOME_VFS_ERROR_BAD_PARAMETERS);

	errBOOLEAN=RtlTimeToSecondsSince1970(Time,&SecondsSince1970);
	g_assert(errBOOLEAN==TRUE);

	timespec->tv_sec=SecondsSince1970;

	/* #define TICKSPERSEC 10000000 */

	/* Some problem with the remainer? */
#if TICKSTO1970%TICKSPERSEC
#error "Unsupported: TICKSTO1970"
#endif

#if TICKSPERSEC<1000000000 && !(1000000000%TICKSPERSEC)
	timespec->tv_nsec=(Time->QuadPart%TICKSPERSEC) * (1000000000/TICKSPERSEC);
#else
#error "Unsupported: TICKSPERSEC"
#endif

	return GNOME_VFS_OK;
}
