/* $Id: media.c,v 1.15 2003/07/13 19:30:18 short Exp $
 * captive media (cdrom/disk) emulation for reactos
 * Copyright (C) 2002 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "media.h"	/* self */
#include "captive/storage.h"	/* self */
#include <glib/gmessages.h>
#include "reactos/ddk/class2.h"	/* for DEVICE_EXTENSION */
#include "reactos/ddk/status.h"
#include "reactos/ddk/iofuncs.h"	/* for IoIsErrorUserInduced() */
#include "captive/macros.h"
#include "reactos/ddk/mmfuncs.h"	/* for MmGetMdlByteCount() */
#include "captive/unicode.h"
#include <glib/gmacros.h>
#include "reactos/structs.h"	/* for PREVENT_MEDIA_REMOVAL */
#include "captive/options.h"


GIOChannel *captive_image_iochannel;
guint64 captive_image_size;


static gboolean validate_DeviceObject(DEVICE_OBJECT *DeviceObject)
{
DEVICE_EXTENSION *DeviceExtension;
DISK_GEOMETRY *DiskGeometry;
IO_SCSI_CAPABILITIES *PortCapabilities;
struct captive_DriverObject *captive_DriverObject;

	g_return_val_if_fail(DeviceObject!=NULL,FALSE);

	captive_DriverObject=(struct captive_DriverObject *)DeviceObject->DriverObject;
	DeviceExtension=DeviceObject->DeviceExtension;

	DiskGeometry=DeviceExtension->DiskGeometry;
	g_return_val_if_fail(DiskGeometry==&captive_DriverObject->DiskGeometry,FALSE);
	g_return_val_if_fail(DiskGeometry->MediaType==captive_DriverObject->DiskGeometry_check.MediaType,FALSE);
	g_return_val_if_fail(DiskGeometry->TracksPerCylinder==captive_DriverObject->DiskGeometry_check.TracksPerCylinder,FALSE);
	g_return_val_if_fail(DiskGeometry->SectorsPerTrack==captive_DriverObject->DiskGeometry_check.SectorsPerTrack,FALSE);
	g_return_val_if_fail(DiskGeometry->BytesPerSector==captive_DriverObject->DiskGeometry_check.BytesPerSector,FALSE);
	g_return_val_if_fail(DiskGeometry->Cylinders.QuadPart==captive_DriverObject->DiskGeometry_check.Cylinders.QuadPart,FALSE);
	g_return_val_if_fail(DeviceExtension->PartitionLength.QuadPart==(gint64)captive_image_size,FALSE);

	PortCapabilities=DeviceExtension->PortCapabilities;
	g_return_val_if_fail(PortCapabilities==&captive_DriverObject->PortCapabilities,FALSE);
	g_return_val_if_fail(PortCapabilities->Length==captive_DriverObject->PortCapabilities_check.Length,FALSE);
	g_return_val_if_fail(PortCapabilities->MaximumTransferLength==captive_DriverObject->PortCapabilities_check.MaximumTransferLength,FALSE);
	g_return_val_if_fail(PortCapabilities->MaximumPhysicalPages==captive_DriverObject->PortCapabilities_check.MaximumPhysicalPages,FALSE);
	g_return_val_if_fail(PortCapabilities->SupportedAsynchronousEvents==captive_DriverObject->PortCapabilities_check.SupportedAsynchronousEvents,FALSE);
	g_return_val_if_fail(PortCapabilities->AlignmentMask==captive_DriverObject->PortCapabilities_check.AlignmentMask,FALSE);
	g_return_val_if_fail(PortCapabilities->TaggedQueuing==captive_DriverObject->PortCapabilities_check.TaggedQueuing,FALSE);
	g_return_val_if_fail(PortCapabilities->AdapterScansDown==captive_DriverObject->PortCapabilities_check.AdapterScansDown,FALSE);
	g_return_val_if_fail(PortCapabilities->AdapterUsesPio==captive_DriverObject->PortCapabilities_check.AdapterUsesPio,FALSE);

	return TRUE;
}


static NTSTATUS MajorFunction_Irp_finish(DEVICE_OBJECT *DeviceObject,IRP *Irp)
{
NTSTATUS r;

	g_return_val_if_fail(TRUE==validate_DeviceObject(DeviceObject),STATUS_INVALID_PARAMETER);
	g_return_val_if_fail(Irp!=NULL,STATUS_INVALID_PARAMETER);

	/* required for removable media only */
	if (!NT_SUCCESS(Irp->IoStatus.Status) && IoIsErrorUserInduced(Irp->IoStatus.Status)) {
		g_assert(Irp->Tail.Overlay.Thread!=NULL);	/* FIXME: Error should be postponed to first !=NULL Irp later */
		IoSetHardErrorOrVerifyDevice(Irp,DeviceObject);
		Irp->IoStatus.Information=0;	/* may got set during some processing before error occured */
		}

	/* IoCompleteRequest() will do 'IoFreeIrp(Irp);'!
	 * 'IoStatus.Status' must be saved before its invocation!
	 */
	r=Irp->IoStatus.Status;
	IoCompleteRequest(Irp,IO_NO_INCREMENT);
	return r;
}


/* FIXME: We should comply with PDRIVER_DISPATCH prototype but unfortunately
 * CAPTIVE_STDCALL prevents us to do so at least in RedHat gcc-3.2-4 (gcc bug?).
 */
#define MajorFunction_DEVICE_CONTROL ((PDRIVER_DISPATCH)MajorFunction_DEVICE_CONTROL_func)
static NTSTATUS CAPTIVE_STDCALL MajorFunction_DEVICE_CONTROL_func(IN DEVICE_OBJECT *DeviceObject,IN IRP *Irp)
{
struct captive_DriverObject *captive_DriverObject;
IO_STACK_LOCATION *IrpStack;

	g_return_val_if_fail(TRUE==validate_DeviceObject(DeviceObject),STATUS_INVALID_PARAMETER);
	g_return_val_if_fail(Irp!=NULL,STATUS_INVALID_PARAMETER);

	captive_DriverObject=(struct captive_DriverObject *)DeviceObject->DriverObject;

	Irp->IoStatus.Information=0;	/* request-specific, may get overriden later */
	IrpStack=IoGetCurrentIrpStackLocation(Irp);
	g_assert(IrpStack->MajorFunction==IRP_MJ_DEVICE_CONTROL);
	g_assert(IrpStack->MinorFunction==0);

	switch (IrpStack->Parameters.DeviceIoControl.IoControlCode) {

		case IOCTL_CDROM_GET_LAST_SESSION:
			/* Nothing interesting to see, move along.
			 * FIXME: This call is somehow multisession related - must we care?
			 */
			/* PASSTHRU */

		case IOCTL_CDROM_READ_TOC: {
CDROM_TOC *CdromToc;

			if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength<sizeof(CDROM_TOC)) {
				Irp->IoStatus.Status=STATUS_BUFFER_TOO_SMALL;
				Irp->IoStatus.Information=sizeof(CDROM_TOC);
				g_assert_not_reached();
				goto done;
				}
			CdromToc=(CDROM_TOC *)Irp->AssociatedIrp.SystemBuffer;
			CAPTIVE_MEMZERO(CdromToc);
			CdromToc->Length[0]=((sizeof(*CdromToc)-2)>>0U)&0xFFU;	/* little-endian */
			CdromToc->Length[1]=((sizeof(*CdromToc)-2)>>8U)&0xFFU;
			CdromToc->FirstTrack=0;	/* one track; TOC_LAST_TRACK does not count */
			CdromToc->LastTrack =0;	/* one track; TOC_LAST_TRACK does not count */
			CdromToc->TrackData[0].Control=TOC_DATA_TRACK;
			CdromToc->TrackData[0].Adr=0;	/* Q-subchannel subinfo */
			CdromToc->TrackData[0].TrackNumber=0;
			CdromToc->TrackData[0].Address[0]=0>>24U;	/* LBA offset; big-endian */
			CdromToc->TrackData[0].Address[1]=0>>16U;
			CdromToc->TrackData[0].Address[2]=0>> 8U;
			CdromToc->TrackData[0].Address[3]=0>> 0U;
			CdromToc->TrackData[1].Control=0;
			CdromToc->TrackData[1].Adr=0;	/* Q-subchannel subinfo */
			CdromToc->TrackData[1].TrackNumber=TOC_LAST_TRACK;
			/* FIXME: should we put the captive_image_size to TOC_LAST_TRACK? */
			CdromToc->TrackData[1].Address[0]=(captive_image_size/512)>>24U;	/* LBA offset; big-endian */
			CdromToc->TrackData[1].Address[1]=(captive_image_size/512)>>16U;
			CdromToc->TrackData[1].Address[2]=(captive_image_size/512)>> 8U;
			CdromToc->TrackData[1].Address[3]=(captive_image_size/512)>> 0U;

			Irp->IoStatus.Information=sizeof(DISK_GEOMETRY);
			Irp->IoStatus.Status=STATUS_SUCCESS;
			} break;

		case IOCTL_CDROM_GET_DRIVE_GEOMETRY:
		case IOCTL_DISK_GET_DRIVE_GEOMETRY:
			if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength<sizeof(DISK_GEOMETRY)) {
				Irp->IoStatus.Status=STATUS_BUFFER_TOO_SMALL;
				Irp->IoStatus.Information=sizeof(DISK_GEOMETRY);
				g_assert_not_reached();
				goto done;
				}
			*(DISK_GEOMETRY *)Irp->AssociatedIrp.SystemBuffer=captive_DriverObject->DiskGeometry;
			Irp->IoStatus.Information=sizeof(DISK_GEOMETRY);
			Irp->IoStatus.Status=STATUS_SUCCESS;
			break;

		case IOCTL_CDROM_CHECK_VERIFY:
		case IOCTL_DISK_CHECK_VERIFY:
			if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength) {
				if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength<sizeof(ULONG)) {
					Irp->IoStatus.Status=STATUS_BUFFER_TOO_SMALL;
					Irp->IoStatus.Information=sizeof(ULONG);
					g_assert_not_reached();
					goto done;
					}
				*(ULONG *)Irp->AssociatedIrp.SystemBuffer=0;	/* MediaChangeCount */
				Irp->IoStatus.Information=sizeof(ULONG);
				}
			else {
				Irp->IoStatus.Information=0;
				}
			Irp->IoStatus.Status=STATUS_SUCCESS;
			break;

		case IOCTL_SCSI_GET_CAPABILITIES:
			if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength<sizeof(IO_SCSI_CAPABILITIES)) {
				Irp->IoStatus.Status=STATUS_BUFFER_TOO_SMALL;
				Irp->IoStatus.Information=sizeof(IO_SCSI_CAPABILITIES);
				g_assert_not_reached();
				goto done;
				}
			*(IO_SCSI_CAPABILITIES *)Irp->AssociatedIrp.SystemBuffer=captive_DriverObject->PortCapabilities;
			Irp->IoStatus.Information=sizeof(IO_SCSI_CAPABILITIES);
			Irp->IoStatus.Status=STATUS_SUCCESS;
			break;

		case IOCTL_DISK_GET_PARTITION_INFO: {
PARTITION_INFORMATION *PartitionInformation;

			if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength<sizeof(PARTITION_INFORMATION)) {
				Irp->IoStatus.Status=STATUS_BUFFER_TOO_SMALL;
				Irp->IoStatus.Information=sizeof(PARTITION_INFORMATION);
				g_assert_not_reached();
				goto done;
				}
			PartitionInformation=(PARTITION_INFORMATION *)Irp->AssociatedIrp.SystemBuffer;
			PartitionInformation->StartingOffset.QuadPart=0;
			PartitionInformation->PartitionLength.QuadPart=captive_image_size;	/* unit=bytes */
			PartitionInformation->HiddenSectors=0;	/* FIXME: real image disk offset */
			PartitionInformation->PartitionNumber=1;
			PartitionInformation->PartitionType=0;	/* FIXME: meaning? */
			PartitionInformation->BootIndicator=TRUE;
			PartitionInformation->RecognizedPartition=TRUE;
			PartitionInformation->RewritePartition=FALSE;	/* FIXME: meaning? */
			Irp->IoStatus.Information=sizeof(PARTITION_INFORMATION);
			Irp->IoStatus.Status=STATUS_SUCCESS;
			} break;

		case IOCTL_DISK_GET_PARTITION_INFO_EX: {
PARTITION_INFORMATION_EX *PartitionInformationEx;

			if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength<sizeof(PARTITION_INFORMATION_EX)) {
				Irp->IoStatus.Status=STATUS_BUFFER_TOO_SMALL;
				Irp->IoStatus.Information=sizeof(PARTITION_INFORMATION_EX);
				g_assert_not_reached();
				goto done;
				}
			PartitionInformationEx=(PARTITION_INFORMATION_EX *)Irp->AssociatedIrp.SystemBuffer;
			PartitionInformationEx->PartitionStyle=PARTITION_STYLE_RAW;	/* not MBR or GPT */
			PartitionInformationEx->StartingOffset.QuadPart=0;
			PartitionInformationEx->PartitionLength.QuadPart=captive_image_size;	/* unit=bytes */
			PartitionInformationEx->PartitionNumber=0;	/* FIXME: what number? */
			PartitionInformationEx->RewritePartition=FALSE;	/* FIXME: meaning? */
			Irp->IoStatus.Information=sizeof(PARTITION_INFORMATION_EX);
			Irp->IoStatus.Status=STATUS_SUCCESS;
			} break;

		case IOCTL_STORAGE_GET_HOTPLUG_INFO:
			Irp->IoStatus.Status=STATUS_NOT_SUPPORTED;
			break;

		case IOCTL_DISK_IS_WRITABLE:
			Irp->IoStatus.Information=0;
			switch (captive_options->rwmode) {
				case CAPTIVE_OPTION_RWMODE_RO:    Irp->IoStatus.Status=STATUS_MEDIA_WRITE_PROTECTED; break;
				case CAPTIVE_OPTION_RWMODE_BLIND: Irp->IoStatus.Status=STATUS_SUCCESS;               break;
				case CAPTIVE_OPTION_RWMODE_RW:    Irp->IoStatus.Status=STATUS_SUCCESS;               break;
				default: g_assert_not_reached();
				}
			break;

		case IOCTL_DISK_MEDIA_REMOVAL: {
PREVENT_MEDIA_REMOVAL *PreventMediaRemoval;

			/* FIXME: ntfs.sys calls this function with empty buffer; dunno what it means.
			 */
			if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength==0) {
				g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"%s: IOCTL_DISK_MEDIA_REMOVAL with empty buffer",G_STRLOC);
				Irp->IoStatus.Information=0;
				Irp->IoStatus.Status=STATUS_SUCCESS;
				break;
				}
			if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength<sizeof(PREVENT_MEDIA_REMOVAL)) {
				Irp->IoStatus.Status=STATUS_BUFFER_TOO_SMALL;
				Irp->IoStatus.Information=sizeof(PREVENT_MEDIA_REMOVAL);
				g_assert_not_reached();
				goto done;
				}
			PreventMediaRemoval=(PREVENT_MEDIA_REMOVAL *)Irp->AssociatedIrp.SystemBuffer;
			g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"%s: IOCTL_DISK_MEDIA_REMOVAL now %s",
					G_STRLOC,(PreventMediaRemoval->PreventMediaRemoval ? "TRUE" : "FALSE"));

			Irp->IoStatus.Information=sizeof(PREVENT_MEDIA_REMOVAL);
			Irp->IoStatus.Status=STATUS_SUCCESS;
			} break;

		case IOCTL_DISK_CONTROLLER_NUMBER: {
DISK_CONTROLLER_NUMBER *DiskControllerNumber;

			if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength<sizeof(DISK_CONTROLLER_NUMBER)) {
				Irp->IoStatus.Status=STATUS_BUFFER_TOO_SMALL;
				Irp->IoStatus.Information=sizeof(DISK_CONTROLLER_NUMBER);
				g_assert_not_reached();
				goto done;
				}
			DiskControllerNumber=(DISK_CONTROLLER_NUMBER *)Irp->AssociatedIrp.SystemBuffer;
			DiskControllerNumber->ControllerNumber=0;	/* FIXME: based on 0 or 1? */
			DiskControllerNumber->DiskNumber=0;	/* FIXME: based on 0 or 1? */

			Irp->IoStatus.Information=sizeof(DISK_CONTROLLER_NUMBER);
			Irp->IoStatus.Status=STATUS_SUCCESS;
			} break;

		case IOCTL_DISK_GET_LENGTH_INFO: {
GET_LENGTH_INFORMATION *GetLengthInformation;

			if (IrpStack->Parameters.DeviceIoControl.OutputBufferLength<sizeof(GET_LENGTH_INFORMATION)) {
				Irp->IoStatus.Status=STATUS_BUFFER_TOO_SMALL;
				Irp->IoStatus.Information=sizeof(GET_LENGTH_INFORMATION);
				g_assert_not_reached();
				goto done;
				}
			GetLengthInformation=(GET_LENGTH_INFORMATION *)Irp->AssociatedIrp.SystemBuffer;
			GetLengthInformation->Length.QuadPart=captive_image_size;

			Irp->IoStatus.Information=sizeof(GET_LENGTH_INFORMATION);
			Irp->IoStatus.Status=STATUS_SUCCESS;
			} break;

		case 0x00664016:
			g_error("%s: Invalid media IOCTL 0x%08lx - by NtfsFixDataError()",G_STRLOC,
					(unsigned long)IrpStack->Parameters.DeviceIoControl.IoControlCode);
			/* FALLTHRU */

		default:
			g_error("%s: Invalid media IOCTL 0x%08lX",G_STRLOC,
					(unsigned long)IrpStack->Parameters.DeviceIoControl.IoControlCode);
			Irp->IoStatus.Status=STATUS_INVALID_DEVICE_REQUEST;
			g_assert_not_reached();
			goto done;
		}
	/* PASSTHRU */

done:	/* 'err:' but we flow here even during success */
	return MajorFunction_Irp_finish(DeviceObject,Irp);
}


struct MajorFunction_READ_WRITE_func_Parameters {
		ULONG Length;
		ULONG Key;
		LARGE_INTEGER ByteOffset;
		};

/* FIXME: We should comply with PDRIVER_DISPATCH prototype but unfortunately
 * CAPTIVE_STDCALL prevents us to do so at least in RedHat gcc-3.2-4 (gcc bug?).
 */
#define MajorFunction_READ_WRITE ((PDRIVER_DISPATCH)MajorFunction_READ_WRITE_func)
static NTSTATUS CAPTIVE_STDCALL MajorFunction_READ_WRITE_func(IN DEVICE_OBJECT *DeviceObject,IN IRP *Irp)
{
IO_STACK_LOCATION *IrpStack;
gpointer buffer=NULL;
GIOStatus erriostatus;
struct captive_DriverObject *captive_DriverObject;
const struct MajorFunction_READ_WRITE_func_Parameters *Parameters=NULL;	/* Prevent: `Parameters' might be used uninitialized */

  g_return_val_if_fail(TRUE==validate_DeviceObject(DeviceObject),STATUS_INVALID_PARAMETER);
	g_return_val_if_fail(Irp!=NULL,STATUS_INVALID_PARAMETER);

	captive_DriverObject=(struct captive_DriverObject *)DeviceObject->DriverObject;

	Irp->IoStatus.Information=0;	/* request-specific, may get overriden later */
	IrpStack=IoGetCurrentIrpStackLocation(Irp);
	g_assert(IrpStack->MajorFunction==IRP_MJ_READ || IrpStack->MajorFunction==IRP_MJ_WRITE);
	g_assert(IrpStack->MinorFunction==0);

#define READ_WRITE_ASSERT_PARAMETERS_OFFSET(struct_a,struct_b,member) \
		g_assert(G_STRUCT_OFFSET(typeof(IrpStack->Parameters.struct_a),member) \
				==G_STRUCT_OFFSET(typeof(IrpStack->Parameters.struct_b),member))
	READ_WRITE_ASSERT_PARAMETERS_OFFSET(Read,Write,Length);
	READ_WRITE_ASSERT_PARAMETERS_OFFSET(Read,Write,Key);
	READ_WRITE_ASSERT_PARAMETERS_OFFSET(Read,Write,ByteOffset);
	READ_WRITE_ASSERT_PARAMETERS_OFFSET(Read,Write,ByteOffset.QuadPart);
#undef READ_WRITE_ASSERT_PARAMETERS_OFFSET

	switch (IrpStack->MajorFunction) {
		case IRP_MJ_READ:  Parameters=(const struct MajorFunction_READ_WRITE_func_Parameters *)&IrpStack->Parameters.Read; break;
		case IRP_MJ_WRITE: Parameters=(const struct MajorFunction_READ_WRITE_func_Parameters *)&IrpStack->Parameters.Write; break;
		default: g_assert_not_reached();
		}

	/* Autodetect 'buffer' as we are !DO_BUFFERED_IO && !DO_DIRECT_IO hybrid */
	if (Irp->UserBuffer) {
		g_assert(buffer==NULL);
		buffer=Irp->UserBuffer;
		}
	/* Forbid both IRP_BUFFERED_IO and IRP_ASSOCIATED_IRP as it should be IMO invalid state. */
	g_assert(!((Irp->Flags & IRP_BUFFERED_IO) && (Irp->Flags & IRP_ASSOCIATED_IRP)));
	if (Irp->Flags & IRP_BUFFERED_IO && Irp->AssociatedIrp.SystemBuffer) {
		g_assert(buffer==NULL);
		buffer=Irp->AssociatedIrp.SystemBuffer;
		}
	if (Irp->MdlAddress) {
		/* See comment at ntoskrnl/io/buildirp.c/IoBuildSynchronousFsdRequestWithMdl()
		 * initialization of 'Irp->UserBuffer'.
		 */
		g_assert(buffer==MmGetSystemAddressForMdl(Irp->MdlAddress) || buffer==NULL);
		g_assert(Parameters->Length<=MmGetMdlByteCount(Irp->MdlAddress));
		buffer=MmGetSystemAddressForMdl(Irp->MdlAddress);
		}
	g_assert(buffer!=NULL);

	g_log(G_LOG_DOMAIN,G_LOG_LEVEL_DEBUG,"%s: %s: ByteOffset=0x%llX,Length=0x%lX",
			G_STRLOC,(IrpStack->MajorFunction==IRP_MJ_READ ? "IRP_MJ_READ" : "IRP_MJ_WRITE"),
			(guint64)Parameters->ByteOffset.QuadPart,(gulong)Parameters->Length);

	erriostatus=g_io_channel_seek_position(captive_image_iochannel,
			Parameters->ByteOffset.QuadPart,	/* offset */
			G_SEEK_SET,	/* type */
			NULL);	/* error */
	g_assert(erriostatus==G_IO_STATUS_NORMAL);

	switch (IrpStack->MajorFunction) {
		case IRP_MJ_READ: {
gsize bytesread;

			erriostatus=g_io_channel_read_chars(captive_image_iochannel,
					buffer,	/* buf */
					Parameters->Length,	/* count */
					&bytesread,	/* bytesread */
					NULL);	/* error */
			g_assert(erriostatus==G_IO_STATUS_NORMAL);
			g_assert(bytesread==Parameters->Length);
			} break;

		case IRP_MJ_WRITE: {
gsize byteswritten;

			erriostatus=g_io_channel_write_chars(captive_image_iochannel,
					buffer,	/* buf */
					Parameters->Length,	/* count */
					&byteswritten,	/* byteswritten */
					NULL);	/* error */
			g_assert(erriostatus==G_IO_STATUS_NORMAL);
			g_assert(byteswritten==Parameters->Length);
			} break;

		default: g_assert_not_reached();
		}

	Irp->IoStatus.Information=Parameters->Length;
	Irp->IoStatus.Status=STATUS_SUCCESS;

	/* PASSTHRU */
/* done: */	/* 'err:' but we flow here even during success */
	return MajorFunction_Irp_finish(DeviceObject,Irp);
}


/* FIXME: We should comply with PDRIVER_DISPATCH prototype but unfortunately
 * CAPTIVE_STDCALL prevents us to do so at least in RedHat gcc-3.2-4 (gcc bug?).
 */
#define MajorFunction_SHUTDOWN ((PDRIVER_DISPATCH)MajorFunction_SHUTDOWN_func)
static NTSTATUS CAPTIVE_STDCALL MajorFunction_SHUTDOWN_func(IN DEVICE_OBJECT *DeviceObject,IN IRP *Irp)
{
IO_STACK_LOCATION *IrpStack;
GIOStatus erriostatus;
struct captive_DriverObject *captive_DriverObject;

  g_return_val_if_fail(TRUE==validate_DeviceObject(DeviceObject),STATUS_INVALID_PARAMETER);
	g_return_val_if_fail(Irp!=NULL,STATUS_INVALID_PARAMETER);

	captive_DriverObject=(struct captive_DriverObject *)DeviceObject->DriverObject;

	IrpStack=IoGetCurrentIrpStackLocation(Irp);
	g_assert(IrpStack->MajorFunction==IRP_MJ_SHUTDOWN || IrpStack->MajorFunction==IRP_MJ_FLUSH_BUFFERS);
	g_assert(IrpStack->MinorFunction==0);

	/* libcaptive is not authorized to shutdown 'captive_image_channel'. */
	erriostatus=g_io_channel_flush(
			captive_image_iochannel,	/* channel */
			NULL);	/* error */
	g_assert(erriostatus==G_IO_STATUS_NORMAL);

	Irp->IoStatus.Information=0;
	Irp->IoStatus.Status=STATUS_SUCCESS;

	return MajorFunction_Irp_finish(DeviceObject,Irp);
}


/* FLUSH_BUFFERS has exactly the functionality for us. */
#define MajorFunction_FLUSH_BUFFERS MajorFunction_SHUTDOWN


/* similiar to drivers/storage/cdrom/cdrom.c/DriverEntry()->...
 * ...->CdromClassCreateDeviceObject()->
 * ->reactos/drivers/storage/class2/class2.c/ScsiClassCreateDeviceObject()
 * We should be driving a lower layer PortDevice but currently we
 * do not provide it, I hope W32 filesystems don't touch it.
 */
NTSTATUS captive_media_DriverEntry(struct captive_DriverObject *captive_DriverObject,PUNICODE_STRING RegistryPath)
{
DRIVER_OBJECT *DriverObject;
DEVICE_OBJECT *DeviceObject;
DEVICE_EXTENSION *DeviceExtension;
NTSTATUS err;

	g_return_val_if_fail(captive_DriverObject!=NULL,STATUS_INVALID_PARAMETER);
	g_return_val_if_fail(RegistryPath!=NULL,STATUS_INVALID_PARAMETER);

	DriverObject=(DRIVER_OBJECT *)captive_DriverObject;

	err=IoCreateDevice(
			DriverObject,	/* DriverObject */
			sizeof(DEVICE_EXTENSION),	/* DeviceExtensionSize; additional storage not used */
			captive_utf8_to_UnicodeString_alloca(captive_DriverObject->DeviceName_utf8),	/* DeviceName */
			captive_DriverObject->DeviceType,	/* DeviceType */
			captive_DriverObject->DeviceCharacteristics,	/* DeviceCharacteristics */
			FALSE,	/* Exclusive */
			&DeviceObject);	/* DeviceObject */
	g_return_val_if_fail(NT_SUCCESS(err),FALSE);

	/* Currently we are !DO_BUFFERED_IO && !DO_DIRECT_IO and we must solve
	 * being called with 'Irp->UserBuffer', 'Irp->AssociatedIrp.SystemBuffer' or 'Irp->MdlAddress'.
	 */
	/* should be left from IoCreateDevice(DeviceCharacteristics) above: */
	if (captive_DriverObject->DeviceCharacteristics & FILE_REMOVABLE_MEDIA)
		g_assert(DeviceObject->Characteristics & FILE_REMOVABLE_MEDIA);
	/* ignored: DeviceObject->StackSize */
	/* ignored: DeviceObject->AlignmentRequirement */

	/* from reactos/drivers/storage/scsiport/scsiport.c/ScsiPortCreatePortDevice() */
	captive_DriverObject->PortCapabilities.Length=sizeof(captive_DriverObject->PortCapabilities);
	captive_DriverObject->PortCapabilities.MaximumTransferLength=0x10000;	/* 64KB */
	g_assert((captive_DriverObject->PortCapabilities.MaximumTransferLength%PAGE_SIZE)==0);
	captive_DriverObject->PortCapabilities.MaximumPhysicalPages=captive_DriverObject->PortCapabilities.MaximumTransferLength/PAGE_SIZE;
	captive_DriverObject->PortCapabilities.SupportedAsynchronousEvents=0;
	captive_DriverObject->PortCapabilities.AlignmentMask=1;	/* no alignment required by us; speced as "integer multiple" */
	captive_DriverObject->PortCapabilities.TaggedQueuing=FALSE;
	captive_DriverObject->PortCapabilities.AdapterScansDown=FALSE;
	captive_DriverObject->PortCapabilities.AdapterUsesPio=TRUE;
	captive_DriverObject->PortCapabilities_check=captive_DriverObject->PortCapabilities;	/* for g_assert() checking against foreign modifications */

	DeviceExtension=DeviceObject->DeviceExtension;
	DeviceExtension->MediaChangeCount=0;
	DeviceExtension->PhysicalDevice=DeviceObject;	/* no real PhysicalDeviceObject */
	DeviceExtension->LockCount=0;
	DeviceExtension->DeviceNumber=0;	/* corresponds to the # in "\\Device\\CdRom0" */
	/* ignored DeviceExtension->PortDeviceObject
	 * as we are the final driver and we don't have any PortDeviceObject
	 */
	DeviceExtension->PortCapabilities=&captive_DriverObject->PortCapabilities;
	DeviceExtension->StartingOffset.QuadPart=0;
	DeviceExtension->PartitionLength.QuadPart=captive_image_size;
	DeviceExtension->PortNumber=0;
	DeviceExtension->PathId=0;
	DeviceExtension->TargetId=0;
	DeviceExtension->Lun=0;

	/* expect 'captive_DriverObject->DiskGeometry.MediaType' */
	captive_DriverObject->DiskGeometry.TracksPerCylinder=64;
	captive_DriverObject->DiskGeometry.SectorsPerTrack=32;
	/* expect 'captive_DriverObject->DiskGeometry.BytesPerSector' */
	captive_DriverObject->DiskGeometry.Cylinders.QuadPart=captive_image_size
			/captive_DriverObject->DiskGeometry.BytesPerSector
			/captive_DriverObject->DiskGeometry.SectorsPerTrack
			/captive_DriverObject->DiskGeometry.TracksPerCylinder;
	/* 'DeviceExtension->DiskGeometry' is NULL! */
	captive_DriverObject->DiskGeometry_check=captive_DriverObject->DiskGeometry;	/* for g_assert() checking against foreign modifications */
	DeviceExtension->DiskGeometry=&captive_DriverObject->DiskGeometry;

	DriverObject->MajorFunction[IRP_MJ_DEVICE_CONTROL]=MajorFunction_DEVICE_CONTROL;
	DriverObject->MajorFunction[IRP_MJ_READ          ]=MajorFunction_READ_WRITE;
	DriverObject->MajorFunction[IRP_MJ_WRITE         ]=MajorFunction_READ_WRITE;
	DriverObject->MajorFunction[IRP_MJ_SHUTDOWN      ]=MajorFunction_SHUTDOWN;
	DriverObject->MajorFunction[IRP_MJ_FLUSH_BUFFERS ]=MajorFunction_FLUSH_BUFFERS;

	return STATUS_SUCCESS;
}
