/* $Id: relastblock.c,v 1.2 2003/12/21 21:28:31 short Exp $
 * Workaround Linux kernel bug wrt accessing last device block for libcaptive
 * Copyright (C) 2003 Jan Kratochvil <project-captive@jankratochvil.net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; exactly version 2 of June 1991 is required
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */


#include "config.h"

#include "iounixchannel.h"	/* self */
#include <glib/gmessages.h>
#include <glib/giochannel.h>
#include <glib/gtypes.h>
#include <unistd.h>
#include <fcntl.h>
#include "size.h"
#include <linux/hdreg.h>
#include <sys/ioctl.h>
#include "captive/macros.h"
#include <ctype.h>
#include "../client/lib.h"	/* for captive_giochannel_setup(); FIXME: pathname */
#include "../client/giochannel-subrange.h"
#include "captive/storage.h"	/* for captive_giochannel_size() */


/* Config */
#define ENSURE_BLOCK_SIZE 0x200	/* size of the NTFS 'superblock backup' */



gboolean start_bytes_ioctl_detect(int fd,guint64 *start_bytes_ioctlp)
{
struct hd_geometry hd_geometry;
#ifdef HDIO_GETGEO_BIG
struct hd_big_geometry hd_big_geometry;
#endif
unsigned long start_sectors_ioctl;

	/* 'start' field is always 'unsigned long'.
	 * 'HDIO_GETGEO_BIG' is supported if 'HDIO_GETGEO' gets deprecated.
	 */
	if (0)
		/* nop */;
#ifdef HDIO_GETGEO_BIG
	else if (!ioctl(fd,HDIO_GETGEO_BIG,&hd_big_geometry))
		start_sectors_ioctl=hd_big_geometry.start;
#endif
	else if (!ioctl(fd,HDIO_GETGEO,&hd_geometry))
		start_sectors_ioctl=hd_geometry.start;
	else
		return FALSE;

	*start_bytes_ioctlp=((guint64)start_sectors_ioctl)*0x200;
	return TRUE;
}

static gboolean check_last_block(GIOChannel *iochannel,guint64 iochannel_size)
{
GIOStatus erriostatus;
gchar buf[ENSURE_BLOCK_SIZE];
gsize bufgot;

	g_return_val_if_fail(iochannel!=NULL,FALSE);
	g_return_val_if_fail(iochannel_size>0,FALSE);

	erriostatus=g_io_channel_seek_position(iochannel,iochannel_size-ENSURE_BLOCK_SIZE,G_SEEK_SET,
			NULL);	/* error */
	g_assert(erriostatus==G_IO_STATUS_NORMAL);

	erriostatus=g_io_channel_read_chars(iochannel,
			buf, /* buf */
			ENSURE_BLOCK_SIZE,	/* count */
			&bufgot,	/* bytes_read */
			NULL);	/* error */
	/* During read on the end boundary of Linux kernel block device we will
	 * get GNOME_VFS_ERROR_IO at least from linux-kernel-2.4.19-ac4
	 * which will get mapped to G_IO_STATUS_ERROR by captive_gnomevfs_giognomevfs_io_read().
	 */
	g_assert(0
			|| (bufgot==ENSURE_BLOCK_SIZE && erriostatus==G_IO_STATUS_NORMAL)
			|| (bufgot==0                 && erriostatus==G_IO_STATUS_EOF)
			|| (bufgot==0                 && erriostatus==G_IO_STATUS_ERROR));

	return (erriostatus==G_IO_STATUS_NORMAL);
}

/* No new reference is created; unref the result the same way as the input. */
GIOChannel *captive_storage_relastblock(GIOChannel *iochannel)
{
int fd;
guint64 size_bytes_ioctl;
guint64 start_bytes_ioctl;
char linkbuf[0x1000],*linknum;
int linkgot;
const gchar *iochannel_unix_new_mode;
const gchar *linkread_pathname;
const gchar *slashpart_prefix;
gboolean errbool;
GIOFlags iochannel_flags;
GIOChannel *iochannel_unix_new;
int iochannel_unix_new_fd;
guint64 unix_new_start_bytes_ioctl,unix_new_size_bytes_ioctl;
GIOChannel *iochannel_subrange_new;

	g_return_val_if_fail(iochannel!=NULL,NULL);

	if (-1==(fd=captive_iounixchannel_get_fd(iochannel)))
		return iochannel;

	if (ENSURE_BLOCK_SIZE>(size_bytes_ioctl=captive_giochannel_size_ioctl(iochannel)))
		return iochannel;

	g_return_val_if_fail(g_io_channel_get_encoding(iochannel)==NULL,0);

	if (!start_bytes_ioctl_detect(fd,&start_bytes_ioctl))
		return iochannel;

	if (check_last_block(iochannel,size_bytes_ioctl))
		return iochannel;

	linkread_pathname=captive_printf_alloca("/proc/self/fd/%d",fd);
	linkgot=readlink(
			linkread_pathname,	/* path */
			linkbuf,	/* buf */
			sizeof(linkbuf));	/* bufsiz */
	if (!(linkgot>=1 && linkgot<=(int)sizeof(linkbuf)-1))
		g_error(_("Unable to read: %s"),linkread_pathname);
	linkbuf[linkgot]=0;	/* readlink(2) does not '\0'-terminate it */

	for (linknum=linkbuf+linkgot;linknum>linkbuf && isdigit(linknum[-1]);linknum--);
	if (linknum>=linkbuf+linkgot)
		g_error(_("Last block not readable although the link read has no trailing number: %s"),linkbuf);
	*linknum='\0';

	/* /dev/ide/host0/bus0/target0/lun0/part1 -> /dev/ide/host0/bus0/target0/lun0/disc */
	slashpart_prefix="/part";
	if (linknum>linkbuf+strlen(slashpart_prefix) && !strcmp(linknum-strlen(slashpart_prefix),slashpart_prefix))
		strcpy(linknum-strlen(slashpart_prefix),"/disc");

	/* /dev/ataraid/d0p1 -> /dev/ataraid/d0 */
	if (linknum>=linkbuf+2 && linknum[-1]=='p' && isdigit(linknum[-2]))
		*--linknum='\0';

	iochannel_flags=g_io_channel_get_flags(iochannel);
	switch (iochannel_flags & (G_IO_FLAG_IS_READABLE|G_IO_FLAG_IS_WRITEABLE)) {
		case G_IO_FLAG_IS_READABLE:
			iochannel_unix_new_mode="r";
			break;
		case G_IO_FLAG_IS_READABLE|G_IO_FLAG_IS_WRITEABLE:
			iochannel_unix_new_mode="r+";
			break;
		default: g_assert_not_reached();
		}

	if (!(iochannel_unix_new=g_io_channel_new_file(
			linkbuf,	/* filename */
			iochannel_unix_new_mode,	/* mode */
			NULL)))	/* error */
		g_error(_("Parent partition \"%s\" not readable by mode \"%s\""),linkbuf,iochannel_unix_new_mode);

	/* 'iochannel_unix_new' sanity checks: */
	iochannel_unix_new_fd=captive_iounixchannel_get_fd(iochannel_unix_new);
	g_assert(iochannel_unix_new_fd>=0);
	errbool=start_bytes_ioctl_detect(iochannel_unix_new_fd,&unix_new_start_bytes_ioctl);
	g_assert(errbool==TRUE);
	g_assert(unix_new_start_bytes_ioctl==0);
	unix_new_size_bytes_ioctl=captive_giochannel_size_ioctl(iochannel_unix_new);
	g_assert(unix_new_size_bytes_ioctl>0);
	if (!(unix_new_size_bytes_ioctl>=start_bytes_ioctl+size_bytes_ioctl))
		g_error(_("Partition last block inaccessible and partition table incorrect: disc size %llu < part start %llu + part size %llu"),
		(unsigned long long)unix_new_size_bytes_ioctl,
		(unsigned long long)start_bytes_ioctl,
		(unsigned long long)size_bytes_ioctl);

	iochannel_subrange_new=(GIOChannel *)captive_giochannel_subrange_new(iochannel_unix_new,
			start_bytes_ioctl,	/* start */
			start_bytes_ioctl+size_bytes_ioctl);	/* end */
	g_assert(iochannel_subrange_new!=NULL);

	g_io_channel_unref(iochannel_unix_new);	/* now reffed by 'iochannel_subrange_new' */

	captive_giochannel_setup(iochannel_subrange_new);
	g_io_channel_unref(iochannel);

	if (!check_last_block(iochannel_subrange_new,size_bytes_ioctl))
		g_error(_("Last block still not readable for the subrange of: %s"),linkbuf);

	if (size_bytes_ioctl!=captive_giochannel_size(iochannel_subrange_new))
		g_error(_("Invalid size of the subranged GIOChannel of: %s"),linkbuf);

	return iochannel_subrange_new;
}
